package net.covers1624.coffeegrinder.type;

import net.covers1624.quack.collection.ColUtils;
import net.covers1624.quack.collection.FastStream;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.Objects;

/**
 * Created by covers1624 on 23/7/21.
 */
public class IntersectionType extends ReferenceType {

    public final @Nullable ReferenceType baseType; // ClassType or TypeParameter
    private final List<ClassType> interfaces;

    public IntersectionType(@Nullable ReferenceType baseType, List<ClassType> interfaces) {
        assert baseType == null || !TypeSystem.isInterface(baseType);
        assert baseType != null || interfaces.size() > 1;
        assert !(baseType instanceof IntersectionType);
        assert !interfaces.isEmpty();
        assert ColUtils.allMatch(interfaces, TypeSystem::isInterface);
        assert FastStream.of(interfaces).map(ClassType::getDeclaration).distinct().count() == interfaces.size();
        this.baseType = baseType;
        this.interfaces = List.copyOf(interfaces);
    }

    @Override
    public String getName() {
        return getDirectSuperTypes().map(AType::getName).join(" & ");
    }

    @Override
    public String getFullName() {
        return getDirectSuperTypes().map(AType::getFullName).join(" & ");
    }

    @Override
    public String toString() {
        return getFullName();
    }

    @Override
    public ReferenceType getSuperType() {
        return baseType != null ? baseType : interfaces.get(0);
    }

    public List<ClassType> getInterfaces() {
        return interfaces;
    }

    @Override
    public FastStream<ReferenceType> getDirectSuperTypes() {
        FastStream<ReferenceType> types = FastStream.of(interfaces);
        if (baseType != null) {
            types = FastStream.of(baseType).concat(types);
        }

        return types;
    }

    @Override
    public boolean mentions(ReferenceType type) {
        return super.mentions(type) || getDirectSuperTypes().anyMatch(e -> e.mentions(type));
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof IntersectionType)) return false;
        IntersectionType other = (IntersectionType) o;

        if (!Objects.equals(baseType, other.baseType)) return false;

        return interfaces.equals(other.interfaces);
    }

    @Override
    public int hashCode() {
        int result = baseType != null ? baseType.hashCode() : 0;
        result = 31 * result + interfaces.hashCode();
        return result;
    }
}
