package net.covers1624.coffeegrinder.type.asm;

import com.google.common.collect.ImmutableList;
import net.covers1624.coffeegrinder.bytecode.AccessFlag;
import net.covers1624.coffeegrinder.type.*;
import net.covers1624.coffeegrinder.type.ClassType.DeclType;
import net.covers1624.coffeegrinder.type.accessors.AccessorParser;
import net.covers1624.coffeegrinder.type.accessors.SyntheticAccessor;
import net.covers1624.coffeegrinder.util.EnumBitSet;
import net.covers1624.coffeegrinder.util.Util;
import net.covers1624.coffeegrinder.util.asm.TypeParameterParser;
import net.covers1624.coffeegrinder.util.resolver.CachedClassNode;
import net.covers1624.quack.collection.FastStream;
import org.jetbrains.annotations.Nullable;
import org.objectweb.asm.Type;
import org.objectweb.asm.tree.MethodNode;
import org.objectweb.asm.tree.ParameterNode;

import java.util.List;
import java.util.Optional;
import java.util.function.Supplier;

import static java.util.Collections.singletonList;
import static net.covers1624.quack.util.SneakyUtils.unsafeCast;

/**
 * Created by covers1624 on 14/4/21.
 */
public class AsmMethod extends Method {

    private final TypeResolver typeResolver;
    private final AsmClass owner;
    private final MethodNode mNode;

    private final boolean isConstructor;

    private final Type desc;
    private final AType returnType;

    private final EnumBitSet<AccessFlag> accessFlags;

    private final Supplier<MethodNode> fullNode;
    private final List<Parameter> parameters;
    private final List<ReferenceType> exceptions;
    private final List<AsmTypeParameter> typeParameters;

    @Nullable
    private final SignatureInfo signatureInfo;

    private final AnnotationSupplier annotationSupplier;
    private final Supplier<Optional<Object>> annotationDefault;

    private final Supplier<Optional<SyntheticAccessor>> accessor;
    private final Supplier<Method> raw;

    AsmMethod(TypeResolver typeResolver, AsmClass owner, MethodNode mNode) {
        this.typeResolver = typeResolver;
        this.owner = owner;
        this.mNode = mNode;

        //TODO move these constants somewhere
        isConstructor = mNode.name.equals("<init>");

        desc = Type.getMethodType(mNode.desc);
        returnType = typeResolver.resolveType(desc.getReturnType());

        accessFlags = AccessFlag.unpackMethod(mNode.access);

        CachedClassNode cNode = owner.getNode();
        fullNode = Util.singleMemoize(() -> cNode.findFullMethod(mNode.name, mNode.desc));

        typeParameters = mNode.signature != null ? TypeParameterParser.parse(mNode.signature, this) : List.of();
        signatureInfo = mNode.signature == null ? null : parseSignatureInfo();

        parameters = parseParameters();
        exceptions = FastStream.of(mNode.exceptions)
                .<ReferenceType>map(typeResolver::resolveClass)
                .toImmutableList();

        assert signatureInfo == null
                || signatureInfo.exceptions.isEmpty()                    // Signature info can be empty.
                || exceptions.size() == signatureInfo.exceptions.size(); // But must be the same size if it exists.

        annotationSupplier = new AnnotationSupplier(typeResolver,
                Util.safeConcat(mNode.visibleAnnotations, mNode.invisibleAnnotations),
                Util.safeConcat(mNode.visibleTypeAnnotations, mNode.invisibleTypeAnnotations)
        );

        annotationDefault = Util.singleMemoize(() -> {
            if (mNode.annotationDefault == null) return Optional.empty();

            return Optional.of(AnnotationParser.processAnnotationDefault(typeResolver, mNode.annotationDefault));
        });

        accessor = Util.singleMemoize(() -> {
            if (!isSynthetic()) return Optional.empty();

            return Optional.ofNullable(AccessorParser.parseAccessor(typeResolver, getNode()));
        });
        raw = Util.singleMemoize(() -> signatureInfo != null ? new RawMethod(this, returnType, exceptions) : this);
    }

    @Nullable
    private SignatureInfo parseSignatureInfo() {
        MethodSignatureParser parser = MethodSignatureParser.parse(typeResolver, this, mNode.signature);

        SignatureInfo signatureInfo = new SignatureInfo(
                parser.getReturnType(),
                parser.getParameters(),
                parser.getExceptions()
        );

        if (hasTypeParameters() || signatureInfo.isGeneric()) {
            return signatureInfo;
        }
        return null;
    }

    private List<Parameter> parseParameters() {
        ImmutableList.Builder<Parameter> builder = ImmutableList.builder();
        Type[] args = desc.getArgumentTypes();

        // With Javac, Synthetic parameters are not classified as 'formal parameters' and thus are not present
        // in the method Signature, and their indexes are skipped when Annotations are present.
        // Synthetic parameters include:
        //  - (beginning) Outer this instance for non-static inner class constructors
        //  - (beginning) Outer this instance for local class constructors inside non-static functions
        //  - (beginning) Enum index and name constructor parameters.
        //  - (End) Local variable bindings into local classes.
        // Javac will generate Signatures for any method which has extended type information that cannot be represented
        // in the descriptor.
        // This is used by both generic signature information and Parameter annotations who both only have indexes
        // into the formal parameter list of a method.
        int formalOffset = 0;
        if (isConstructor) {
            if (owner.getDeclType() != DeclType.TOP_LEVEL
                    && !owner.isStatic()
                    && (!owner.getEnclosingMethod().isPresent() || !owner.getEnclosingMethod().get().isStatic())) {
                formalOffset += 1;
            } else if (owner.isEnum()) {
                formalOffset += 2;
            }
        }
        // If there is no signature information, we don't know the length of the formal param list, Assume it's the descriptor length.
        int numFormals = signatureInfo != null ? signatureInfo.parameters.size() : args.length;

        for (int i = 0; i < args.length; i++) {
            // The index this parameter has into the formal parameter list.
            int formalIndex = i >= formalOffset && i < formalOffset + numFormals ? i - formalOffset : -1;

            AType rawType = typeResolver.resolveType(args[i]);
            AType type = signatureInfo != null && formalIndex != -1 ? signatureInfo.parameters.get(formalIndex) : rawType;
            assert TypeSystem.areErasuresEqual(rawType, type);
            ParameterNode pNode = mNode.parameters == null ? null : mNode.parameters.get(i);

            String name = pNode != null ? pNode.name : "p_" + i;
            EnumBitSet<AccessFlag> flags = AccessFlag.unpackParameter(pNode != null ? pNode.access : 0);
            builder.add(new Parameter(
                    formalIndex,
                    name,
                    this,
                    type,
                    rawType,
                    flags,
                    new AnnotationSupplier(typeResolver,
                            Util.safeConcat(safeGet(mNode.visibleParameterAnnotations, formalIndex), safeGet(mNode.invisibleParameterAnnotations, formalIndex)),
                            Util.safeConcat(mNode.visibleTypeAnnotations, mNode.invisibleTypeAnnotations)
                    )
            ));
        }
        return builder.build();
    }

    @Nullable
    private static <T> T safeGet(@Nullable T @Nullable [] arr, int idx) {
        if (arr == null || idx == -1) return null;
        return arr[idx];
    }

    //@formatter:off
    public MethodNode getNode() { return fullNode.get(); }
    @Override public String getName() { return mNode.name; }
    @Override public ClassType getDeclaringClass() { return owner; }
    @Override public EnumBitSet<AccessFlag> getAccessFlags() { return accessFlags; }
    @Override public AType getReturnType() { return signatureInfo != null ? signatureInfo.returnType : returnType; }
    @Override public Type getDescriptor() { return desc; }
    @Override public boolean isConstructor() { return isConstructor; }
    @Override public List<Parameter> getParameters() { return parameters; }
    @Override public List<ReferenceType> getExceptions() { return signatureInfo != null && !signatureInfo.exceptions.isEmpty() ? signatureInfo.exceptions : exceptions; }
    @Override public List<TypeParameter> getTypeParameters() { return unsafeCast(typeParameters); }
    @Override public Method getDeclaration() { return this; }
    @Override public AnnotationSupplier getAnnotationSupplier() { return annotationSupplier; }
    @Override @Nullable public Object getDefaultAnnotationValue() { return annotationDefault.get().orElse(null); }
    @Nullable @Override public SyntheticAccessor getAccessor() { return accessor.get().orElse(null); }
    @Override public Method asRaw() { return raw.get(); }
    //@formatter:on

    private static class SignatureInfo {

        public final AType returnType;
        public final List<AType> parameters;
        public final List<ReferenceType> exceptions;

        private SignatureInfo(AType returnType, List<AType> parameters, List<ReferenceType> exceptions) {
            this.returnType = returnType;
            this.parameters = parameters;
            this.exceptions = exceptions;
        }

        public boolean isGeneric() {
            return isGeneric(singletonList(returnType))
                    || isGeneric(parameters)
                    || isGeneric(exceptions);
        }

        private static boolean isGeneric(List<? extends AType> types) {
            return FastStream.of(types)
                    .anyMatch(SignatureInfo::isGeneric);
        }

        private static boolean isGeneric(AType type) {
            return type instanceof TypeParameter || type instanceof ParameterizedClass
                    || (type instanceof ArrayType && isGeneric(((ArrayType) type).getElementType()));
        }
    }

}
