package net.covers1624.coffeegrinder.bytecode.matching;

import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.insns.Comparison;
import net.covers1624.coffeegrinder.bytecode.insns.LdcNull;
import net.covers1624.coffeegrinder.bytecode.insns.LocalVariable;
import org.jetbrains.annotations.Nullable;

import static net.covers1624.coffeegrinder.bytecode.matching.LoadStoreMatching.matchLoadLocal;

/**
 * Created by covers1624 on 22/11/21.
 */
public class ComparisonMatching {

    /**
     * Matches the given Instruction to a {@link Comparison} instruction.
     *
     * @param insn The Instruction to match.
     * @return The {@link Comparison} or <code>null</code>.
     */
    @Nullable
    public static Comparison matchComparison(@Nullable Instruction insn) {
        if (!(insn instanceof Comparison comparison)) return null;

        return comparison;
    }

    /**
     * Match the given Instruction to a {@link Comparison} of the given kind.
     *
     * @param insn The Instruction to match.
     * @param kind The Kind of comparison.
     * @return The {@link Comparison} or <code>null</code>.
     */
    @Nullable
    public static Comparison matchComparison(@Nullable Instruction insn, Comparison.ComparisonKind kind) {
        Comparison comparison = matchComparison(insn);
        if (comparison == null) return null;

        if (comparison.getKind() != kind) return null;

        return comparison;
    }

    /**
     * Match the given Instruction to a {@link Comparison} of the given kind
     * which loads the given left and right variables.
     *
     * @param insn  The Instruction to match.
     * @param kind  The Kind of comparison.
     * @param left  The Left {@link LocalVariable}.
     * @param right The right {@link LocalVariable}.
     * @return The {@link Comparison} or <code>null</code>.
     */
    @Nullable
    public static Comparison matchComparison(@Nullable Instruction insn, Comparison.ComparisonKind kind, LocalVariable left, LocalVariable right) {
        Comparison comparison = matchComparison(insn, kind);
        if (comparison == null) return null;

        if (matchLoadLocal(comparison.getLeft(), left) == null) return null;
        if (matchLoadLocal(comparison.getRight(), right) == null) return null;

        return comparison;
    }

    /**
     * Matches a {@link Comparison} which checks if the given variable
     * does not equal {@link LdcNull}.
     *
     * @param insn     The Instruction to match.
     * @param variable The variable.
     * @return The {@link Comparison} or <code>null</code>
     */
    @Nullable
    public static Comparison matchNotEqualNull(@Nullable Instruction insn, LocalVariable variable) {
        Comparison comparison = matchComparison(insn, Comparison.ComparisonKind.NOT_EQUAL);
        if (comparison == null) return null;

        if (matchLoadLocal(comparison.getLeft(), variable) == null) return null;
        if (!(comparison.getRight() instanceof LdcNull)) return null;

        return comparison;
    }

    /**
     * Matches a {@link Comparison} which checks if the given variable
     * equals {@link LdcNull}.
     *
     * @param insn     The Instruction to match.
     * @param variable The variable.
     * @return The {@link Comparison} or <code>null</code>
     */
    @Nullable
    public static Comparison matchEqualNull(@Nullable Instruction insn, LocalVariable variable) {
        Comparison comparison = matchComparison(insn, Comparison.ComparisonKind.EQUAL);
        if (comparison == null) return null;

        if (matchLoadLocal(comparison.getLeft(), variable) == null) return null;
        if (!(comparison.getRight() instanceof LdcNull)) return null;

        return comparison;
    }
}
