package net.covers1624.coffeegrinder.bytecode.matching;

import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.insns.Binary;
import net.covers1624.coffeegrinder.bytecode.insns.BinaryOp;
import net.covers1624.coffeegrinder.bytecode.insns.LdcBoolean;
import net.covers1624.coffeegrinder.bytecode.insns.LdcNumber;
import org.jetbrains.annotations.Nullable;

/**
 * Created by covers1624 on 21/7/21.
 */
public class LdcMatching {

    @Nullable
    public static LdcNumber matchLdcNumber(@Nullable Instruction insn) {
        if (!(insn instanceof LdcNumber n)) return null;

        return n;
    }

    /**
     * Matches the provided instruction against an {@link LdcNumber} instruction who's type is an integer.
     *
     * @param insn The instruction to match.
     * @return The {@link LdcNumber} or <code>null</code>.
     */
    @Nullable
    public static LdcNumber matchLdcInt(@Nullable Instruction insn) {
        LdcNumber ldcNumber = matchLdcNumber(insn);
        if (ldcNumber == null) return null;

        if (!(ldcNumber.getValue() instanceof Integer)) return null;
        return ldcNumber;
    }

    /**
     * Matches the provided instruction against an {@link LdcNumber} instruction
     * with the provided value.
     *
     * @param insn  The Instruction to match.
     * @param value The value that must be loaded by the {@link LdcNumber}.
     * @return The {@link LdcNumber} or <code>null</code>.
     */
    @Nullable
    public static LdcNumber matchLdcInt(@Nullable Instruction insn, int value) {
        LdcNumber ldc = matchLdcInt(insn);
        if (ldc == null || ldc.getValue().intValue() != value) return null;

        return ldc;
    }

    /**
     * Matches the provided instruction against an {@link LdcBoolean} instruction.
     *
     * @param insn The instruction to match.
     * @return The {@link LdcBoolean} or <code>null</code>.
     */
    @Nullable
    public static LdcBoolean matchLdcBoolean(@Nullable Instruction insn) {
        if (!(insn instanceof LdcBoolean b)) return null;

        return b;
    }

    /**
     * Matches the provided instruction against an {@link LdcBoolean} instruction
     * with the provided value.
     *
     * @param insn  The Instruction to match.
     * @param value The value that must be loaded by the {@link LdcBoolean}.
     * @return The {@link LdcBoolean} or <code>null</code>.
     */
    @Nullable
    public static LdcBoolean matchLdcBoolean(@Nullable Instruction insn, boolean value) {
        LdcBoolean ldc = matchLdcBoolean(insn);
        if (ldc == null || ldc.getValue() != value) return null;

        return ldc;
    }

    @Nullable
    public static Instruction matchNegation(@Nullable Instruction insn) {
        if (!(insn instanceof Binary binary)) return null;

        if (binary.getOp() != BinaryOp.SUB) return null;

        LdcNumber ldc = matchLdcNumber(binary.getLeft());
        if (ldc == null || ldc.getValue().doubleValue() != 0) return null;

        return binary.getRight();
    }
}
