package net.covers1624.coffeegrinder.bytecode.matching;

import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import net.covers1624.coffeegrinder.type.Field;
import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.Nullable;

/**
 * Created by covers1624 on 19/4/21.
 */
public class LoadStoreMatching {

    /**
     * Matches the given instruction to a Load instruction.
     *
     * @param insn The insn to match against.
     * @return The Load instruction.
     */
    @Nullable
    public static Load matchLoad(@Nullable Instruction insn) {
        if (!(insn instanceof Load l)) return null;

        return l;
    }

    /**
     * Matches the given instruction to a Store instruction.
     *
     * @param insn The insn to match against.
     * @return The Store instruction.
     */
    @Nullable
    public static Store matchStore(@Nullable Instruction insn) {
        if (!(insn instanceof Store s)) return null;

        return s;
    }

    /**
     * Matches the given instruction to a {@link Store} instruction, whose child is a
     * {@link Load} instruction of a specific variable.
     *
     * @param insn     The insn to match against.
     * @param variable The variable to match against.
     * @return The {@link Store} instruction or <code>null</code>
     */
    @Nullable
    public static Store matchStoreLocalLoadLocal(@Nullable Instruction insn, LocalVariable variable) {
        Store store = matchStoreLocal(insn);
        if (store == null) return null;

        Load load = matchLoadLocal(store.getValue(), variable);
        if (load == null) return null;

        return store;
    }

    @Nullable
    public static LocalReference matchLocalRef(@Nullable Instruction insn) {
        if (!(insn instanceof LocalReference r)) return null;

        return r;
    }

    @Nullable
    public static LocalReference matchLocalRef(@Nullable Instruction insn, LocalVariable variable) {
        LocalReference localRef = matchLocalRef(insn);
        if (localRef == null || localRef.variable != variable) return null;
        return localRef;
    }

    /**
     * Matches the given instruction to a Load instruction.
     *
     * @param insn The insn to match against.
     * @return The Load instruction.
     */
    @Nullable
    @Contract ("null->null")
    public static Load matchLoadLocal(@Nullable Instruction insn) {
        Load load = matchLoad(insn);
        if (load == null || !(load.getReference() instanceof LocalReference)) return null;
        return load;
    }

    /**
     * Matches the given instruction to a Load instruction of a specific variable.
     *
     * @param insn     The insn to match against.
     * @param variable The variable to match against.
     * @return The Load instruction.
     */
    @Nullable
    public static Load matchLoadLocal(@Nullable Instruction insn, LocalVariable variable) {
        Load load = matchLoadLocal(insn);
        if (load == null) return null;

        if (load.getVariable() != variable) return null;
        return load;
    }

    /**
     * Matches the given instruction to a Store instruction.
     *
     * @param insn The insn to match against.
     * @return The Store instruction.
     */
    @Nullable
    public static Store matchStoreLocal(@Nullable Instruction insn) {
        Store store = matchStore(insn);
        if (store == null || !(store.getReference() instanceof LocalReference)) return null;
        return store;
    }

    /**
     * Matches the given instruction to a Store instruction of a specific variable.
     *
     * @param insn     The insn to match against.
     * @param variable The variable to match against.
     * @return The Store instruction.
     */
    @Nullable
    public static Store matchStoreLocal(@Nullable Instruction insn, LocalVariable variable) {
        Store store = matchStoreLocal(insn);
        if (store == null) return null;

        if (store.getVariable() != variable) return null;
        return store;
    }

    @Nullable
    public static FieldReference matchFieldRef(@Nullable Instruction insn) {
        if (!(insn instanceof FieldReference r)) return null;

        return r;
    }

    @Nullable
    public static FieldReference matchFieldRef(@Nullable Instruction insn, Field field) {
        FieldReference fieldRef = matchFieldRef(insn);
        if (fieldRef == null) return null;
        if (!matchField(fieldRef.getField(), field)) return null;

        return fieldRef;
    }

    /**
     * Matches the given instruction to a LoadField of the given field.
     *
     * @param insn The instruction to match.
     * @return The LoadField or <code>null</code>.
     */
    @Nullable
    public static Load matchLoadField(@Nullable Instruction insn) {
        Load load = matchLoad(insn);
        if (load == null) return null;
        if (!(load.getReference() instanceof FieldReference)) return null;

        return load;
    }

    /**
     * Matches the given instruction to a LoadField of the given field.
     *
     * @param insn  The instruction to match.
     * @param field The field to match.
     * @return The LoadField or <code>null</code>.
     */
    @Nullable
    public static Load matchLoadField(@Nullable Instruction insn, Field field) {
        Load load = matchLoad(insn);
        if (load == null) return null;
        if (matchFieldRef(load.getReference(), field) == null) return null;

        return load;
    }

    @Nullable
    public static FieldReference matchLoadFieldRef(@Nullable Instruction insn) {
        Load load = matchLoad(insn);
        if (load == null) return null;
        return matchFieldRef(load.getReference());
    }

    @Nullable
    public static FieldReference matchStoreFieldRef(@Nullable Instruction insn) {
        Store store = matchStore(insn);
        if (store == null) return null;
        return matchFieldRef(store.getReference());
    }

    /**
     * Matches the given instruction to a StoreField of the given field.
     *
     * @param insn The instruction to match.
     * @return The StoreField or <code>null</code>.
     */
    @Nullable
    public static Store matchStoreField(@Nullable Instruction insn) {
        Store store = matchStore(insn);
        if (store == null || !(store.getReference() instanceof FieldReference)) return null;

        return store;
    }

    /**
     * Matches the given instruction to a StoreField of the given field.
     *
     * @param insn  The instruction to match.
     * @param field The field to match.
     * @return The StoreField or <code>null</code>.
     */
    @Nullable
    @Contract ("null,_->null")
    public static Store matchStoreField(@Nullable Instruction insn, Field field) {
        Store store = matchStore(insn);
        if (store == null) return null;
        if (matchFieldRef(store.getReference(), field) == null) return null;
        return store;
    }

    public static boolean matchField(Field field1, Field field2) {
        return field1.getName().equals(field2.getName())
                && field1.getDescriptor().equals(field2.getDescriptor())
                // short-circuit a type system resolve if the references are identical
                && (field1.getDeclaringClass().equals(field2.getDeclaringClass()) || field1.equals(field2));
    }

    public static boolean equivalentFieldDescriptors(Field field1, Field field2) {
        return field1.getName().equals(field2.getName())
                && field1.getDescriptor().equals(field2.getDescriptor())
                // short-circuit a type system resolve if the references are identical
                && field1.getDeclaringClass().equals(field2.getDeclaringClass());
    }

    /**
     * Matches the given instruction to an {@link ArrayLen} instruction.
     *
     * @param insn The insn to match against.
     * @return The {@link ArrayLen} instruction.
     */
    @Nullable
    public static ArrayLen matchArrayLen(@Nullable Instruction insn) {
        if (!(insn instanceof ArrayLen r)) return null;

        return r;
    }

    /**
     * Matches the given instruction to an {@link ArrayLen} instruction which
     * contains a {@link Load} of the given variable.
     *
     * @param insn    The insn to match against.
     * @param loadVar The variable loaded by the {@link ArrayLen} target.
     * @return The {@link ArrayLen} instruction.
     */
    @Nullable
    public static ArrayLen matchArrayLenLoad(@Nullable Instruction insn, LocalVariable loadVar) {
        ArrayLen arrayLen = matchArrayLen(insn);
        if (arrayLen == null) return null;
        Load load = matchLoadLocal(arrayLen.getArray(), loadVar);
        if (load == null) return null;

        return arrayLen;
    }

    @Nullable
    public static ArrayElementReference matchElemRef(@Nullable Instruction insn) {
        if (!(insn instanceof ArrayElementReference r)) return null;

        return r;
    }

    @Nullable
    public static ArrayElementReference matchLoadElemRef(@Nullable Instruction insn) {
        Load load = matchLoad(insn);
        if (load == null) return null;

        return matchElemRef(load.getReference());
    }
}
