package net.covers1624.coffeegrinder.bytecode.transform.transformers;

import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import net.covers1624.coffeegrinder.bytecode.matching.IfMatching;
import net.covers1624.coffeegrinder.bytecode.transform.MethodTransformContext;
import net.covers1624.coffeegrinder.bytecode.transform.MethodTransformer;
import net.covers1624.coffeegrinder.type.ClassType;
import net.covers1624.coffeegrinder.type.TypeResolver;
import net.covers1624.coffeegrinder.type.TypeSystem;
import org.jetbrains.annotations.Nullable;
import org.objectweb.asm.Type;

import java.util.List;

import static net.covers1624.coffeegrinder.bytecode.matching.BlockMatching.getBlockOnlyChild;
import static net.covers1624.coffeegrinder.bytecode.matching.BranchLeaveMatching.matchLeave;
import static net.covers1624.coffeegrinder.bytecode.matching.BranchLeaveMatching.matchThrow;
import static net.covers1624.coffeegrinder.bytecode.matching.ComparisonMatching.matchNotEqualNull;
import static net.covers1624.coffeegrinder.bytecode.matching.InvokeMatching.matchInvoke;
import static net.covers1624.coffeegrinder.bytecode.matching.LoadStoreMatching.*;
import static net.covers1624.coffeegrinder.bytecode.matching.TryCatchMatching.matchTryCatch;

/**
 * Handles matching Java 7 - 10 TryWithResources using finally blocks.
 * <p>
 * Created by covers1624 on 17/7/21.
 */
public class LegacyTryWithResourcesTransform implements MethodTransformer {

    private static final Type AUTO_CLOSEABLE = Type.getType(AutoCloseable.class);

    private final ClassType autoCloseableClass;

    public LegacyTryWithResourcesTransform(TypeResolver typeResolver) {
        autoCloseableClass = typeResolver.resolveClass(AUTO_CLOSEABLE);
    }

    @Override
    public void transform(MethodDecl function, MethodTransformContext ctx) {

        boolean done = false;
        while (!done) {
            done = true;
            List<TryFinally> tryFinallys = function.descendantsToList(TryFinally.class);
            for (TryFinally tryFinally : tryFinallys) {
                if (transformTryWithResources(tryFinally, ctx)) {
                    // Whenever we process a try-finally into a TWR, restart.
                    done = false;
                    break;
                }
            }
        }
    }

    private boolean transformTryWithResources(TryFinally tryFinally, MethodTransformContext ctx) {
        // Match the following case:
        // STORE autoCloseable(...)
        // STORE synException(LDC_NULL)
        // TRY_FINALLY {
        //     TRY_CATCH { // Rethrow Try
        //         ...
        //     } catch (java.lang.Throwable ex) { // Rethrow Catch
        //         STORE synException(LOAD ex)
        //         THROW(LOAD ex)
        //     }
        // } finally {
        //     if (autoCloseable != null) {
        //         if (synException != null) {
        //             TRY_CATCH { // Close Try
        //                 INVOKE.VIRTUAL(LOAD autoCloseable) java/lang/AutoCloseable.close()
        //             } catch (java.lang.Throwable ex) {
        //                 INVOKE.VIRTUAL (LOAD synException) java.lang.Throwable.addSuppressed(LOAD ex)
        //             }
        //         } else {
        //             INVOKE.VIRTUAL(LOAD autoCloseable) java/lang/AutoCloseable.close()
        //         }
        //     }
        // }
        // ->
        // TRY_WITH_RESOURCES (STORE autoCloseable(...)) {
        //    ...
        // }

        // Match the Synthetic exception store before the try-finally
        Store synExceptionStore = matchStoreNull(tryFinally.getPrevSiblingOrNull());
        if (synExceptionStore == null) return false;
        LocalVariable synException = synExceptionStore.getVariable();
        if (!synException.isSynthetic() || synException.getLoadCount() != 2 || synException.getStoreCount() != 2) return false;

        // Match the AutoCloseable definition before the try-finally
        Store autoCloseableStore = matchStoreLocal(synExceptionStore.getPrevSiblingOrNull());
        if (autoCloseableStore == null) return false;
        LocalVariable autoCloseable = autoCloseableStore.getVariable();
        if (!TypeSystem.isAssignableTo(autoCloseableStore.getResultType(), autoCloseableClass)) return false;

        // Match Rethrow Try.
        BlockContainer tryFinallyBody = tryFinally.getTryBody();
        if (tryFinallyBody.blocks.size() != 1) return false;
        Block tryFinallyBodyEntryPoint = tryFinallyBody.getEntryPoint();
        if (tryFinallyBodyEntryPoint.instructions.size() != 1 && tryFinallyBodyEntryPoint.instructions.size() != 2) return false;
        TryCatch rethrowTry = matchTryCatch(tryFinallyBodyEntryPoint.getFirstChild());
        if (rethrowTry == null) return false;
        if (rethrowTry.handlers.size() != 1) return false;
        // This container will have a leave as the second instruction, if there is fallthrough content after the TWR.
        if (tryFinallyBodyEntryPoint.instructions.size() == 2) {
            if (matchLeave(tryFinallyBodyEntryPoint.getLastChild(), tryFinallyBody) == null) return false;
        }

        // Match Rethrow Catch.
        TryCatch.TryCatchHandler rethrowCatch = rethrowTry.handlers.first();
        BlockContainer rethrowCatchBody = rethrowCatch.getBody();
        if (rethrowCatchBody.blocks.size() != 1) return false;
        Block rethrowCatchEntry = rethrowCatchBody.getEntryPoint();
        if (rethrowCatchEntry.instructions.size() != 2) return false;

        // Match rethrow Catch body.
        if (matchStoreLocalLoadLocal(rethrowCatchEntry.getFirstChild(), rethrowCatch.getVariable().variable) == null) return false;
        if (matchThrow(rethrowCatchEntry.getLastChild(), rethrowCatch.getVariable().variable) == null) return false;

        // Match finally body.
        BlockContainer finallyBody = tryFinally.getFinallyBody();
        if (finallyBody.blocks.size() != 1) return false;
        Block finallyBodyEntryPoint = finallyBody.getEntryPoint();
        if (finallyBodyEntryPoint.instructions.size() != 2) return false;

        // Match if (autoCloseable != null)
        IfInstruction autoCloseableIf = IfMatching.matchNopFalseIf(finallyBodyEntryPoint.getFirstChild());
        if (autoCloseableIf == null) return false;
        if (matchLeave(autoCloseableIf.getNextSibling(), finallyBody) == null) return false;
        if (matchNotEqualNull(autoCloseableIf.getCondition(), autoCloseable) == null) return false;

        // match if (synException != null)
        IfInstruction synExceptionIf = IfMatching.matchIf(getBlockOnlyChild(autoCloseableIf.getTrueInsn()));
        if (synExceptionIf == null) return false;
        if (matchNotEqualNull(synExceptionIf.getCondition(), synException) == null) return false;

        // Match else block of synExceptionIf
        if (!matchAutoCloseableCloseInvoke(getBlockOnlyChild(synExceptionIf.getFalseInsn()), autoCloseable)) return false;

        // Match then block of synExceptionIf
        TryCatch closeTryCatch = matchTryCatch(getBlockOnlyChild(synExceptionIf.getTrueInsn()));
        if (closeTryCatch == null) return false;
        if (closeTryCatch.handlers.size() != 1) return false;

        // Match close try body.
        BlockContainer closeTCBody = closeTryCatch.getTryBody();
        if (closeTCBody.blocks.size() != 1) return false;
        Block closeTCBodyEntryPoint = closeTCBody.getEntryPoint();
        if (closeTCBodyEntryPoint.instructions.size() != 2) return false;
        if (!matchAutoCloseableCloseInvoke(closeTCBodyEntryPoint.getFirstChild(), autoCloseable)) return false;
        if (matchLeave(closeTCBodyEntryPoint.getLastChild(), closeTCBody) == null) return false;

        // Match close try catch body.
        TryCatch.TryCatchHandler closeCatch = closeTryCatch.handlers.first();
        BlockContainer closeCatchBody = closeCatch.getBody();
        if (closeCatchBody.blocks.size() != 1) return false;
        Block closeCatchBodyEntryPoint = closeCatchBody.getEntryPoint();
        if (closeCatchBodyEntryPoint.instructions.size() != 2) return false;
        Invoke invoke = matchInvoke(closeCatchBodyEntryPoint.getFirstChild(), Invoke.InvokeKind.VIRTUAL, "addSuppressed");
        if (invoke == null) return false;
        if (invoke.getArguments().size() != 1) return false;
        if (matchLoadLocal(invoke.getTarget(), synException) == null) return false;
        if (matchLoadLocal(invoke.getArguments().first(), closeCatch.getVariable().variable) == null) return false;
        if (matchLeave(closeCatchBodyEntryPoint.getLastChild(), closeCatchBody) == null) return false;

        // Lezgo bois we matched a TWR!
        ctx.pushStep("produce try-with-resources");
        tryFinally.replaceWith(new TryWithResources(autoCloseableStore, rethrowTry.getTryBody()));
        synExceptionStore.remove();
        ctx.popStep();
        return true;
    }

    @Nullable
    public static Store matchStoreNull(@Nullable Instruction insn) {
        Store store = matchStoreLocal(insn);
        if (store == null) return null;

        if (!(store.getValue() instanceof LdcNull)) return null;

        return store;
    }

    private boolean matchAutoCloseableCloseInvoke(@Nullable Instruction insn, LocalVariable autoCloseable) {
        Invoke invoke = matchInvoke(insn);
        if (invoke == null) return false;
        if (matchLoadLocal(invoke.getTarget(), autoCloseable) == null) return false;
        if (invoke.getKind() != Invoke.InvokeKind.VIRTUAL && invoke.getKind() != Invoke.InvokeKind.INTERFACE) return false;
        if (!invoke.getMethod().getName().equals("close")) return false;
        return invoke.getMethod().getParameters().isEmpty();
    }
}
