package net.covers1624.coffeegrinder.util.asm;

import com.google.common.collect.ImmutableMap;
import org.objectweb.asm.*;
import org.objectweb.asm.tree.ClassNode;
import org.objectweb.asm.tree.MethodNode;
import org.objectweb.asm.util.*;

import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Created by covers1624 on 19/9/21.
 */
public class OrderedTextifier {

    public static void main(String[] args) throws IOException {
        Path path = Paths.get(String.join(" ", args));
        try (InputStream is = Files.newInputStream(path)) {
            ClassReader reader = new ClassReader(is);
            ClassNode cNode = new ClassNode();
            reader.accept(cNode, 0);
            System.out.println(textify(cNode));
        }
    }

    public static String textify(ClassNode cNode) {
        Textifier textifier = new LabelSettingTextifier();
        cNode.accept(new CustomTraceClassVisitor(cNode, textifier));
        return toString(textifier);
    }

    public static String textify(MethodNode mNode) {
        LabelSettingTextifier textifier = new LabelSettingTextifier();
        textifier.setLabels(extractLabels(mNode));
        mNode.accept(new TraceMethodVisitor(textifier.visitMethod(
                mNode.access,
                mNode.name,
                mNode.desc,
                mNode.signature,
                mNode.exceptions == null ? null : mNode.exceptions.toArray(new String[0])
        )));
        return toString(textifier);
    }

    private static String toString(Textifier textifier) {
        StringWriter stringWriter = new StringWriter();
        PrintWriter printWriter = new PrintWriter(stringWriter);
        textifier.print(printWriter);
        printWriter.flush();
        return stringWriter.toString();
    }

    private static List<Label> extractLabels(MethodNode mNode) {
        if (mNode.instructions.size() == 0) return List.of();

        LabelRange range = LabelRange.extractRange(mNode.instructions);
        if (range == null) return Collections.emptyList();
        return range.range;
    }

    private static class CustomTraceClassVisitor extends ClassVisitor {

        private final Map<String, List<Label>> labels;
        private final Textifier t;

        private CustomTraceClassVisitor(ClassNode cNode, Textifier t) {
            super(Opcodes.ASM9);
            this.t = t;
            ImmutableMap.Builder<String, List<Label>> labelsBuilder = ImmutableMap.builder();
            for (MethodNode method : cNode.methods) {
                labelsBuilder.put(method.name + method.desc, extractLabels(method));
            }
            labels = labelsBuilder.build();
        }

        @Override
        public MethodVisitor visitMethod(int access, String name, String descriptor, String signature, String[] exceptions) {
            LabelSettingTextifier textifier = (LabelSettingTextifier) t.visitMethod(access, name, descriptor, signature, exceptions);
            textifier.setLabels(labels.get(name + descriptor));
            return new TraceMethodVisitor(textifier);
        }

        //@formatter:off
        @Override public void visit(int version, int access, String name, String signature, String superName, String[] interfaces) { t.visit(version, access, name, signature, superName, interfaces); }
        @Override public void visitSource(String source, String debug) { t.visitSource(source, debug); }
        @Override public ModuleVisitor visitModule(String name, int access, String version) { return new TraceModuleVisitor(t.visitModule(name, access, version)); }
        @Override public void visitNestHost(String nestHost) { t.visitNestHost(nestHost); }
        @Override public void visitOuterClass(String owner, String name, String descriptor) { t.visitOuterClass(owner, name, descriptor); }
        @Override public AnnotationVisitor visitAnnotation(String descriptor, boolean visible) { return new TraceAnnotationVisitor(t.visitClassAnnotation(descriptor, visible)); }
        @Override public AnnotationVisitor visitTypeAnnotation(int typeRef, TypePath typePath, String descriptor, boolean visible) { return new TraceAnnotationVisitor(t.visitClassTypeAnnotation(typeRef, typePath, descriptor, visible)); }
        @Override public void visitAttribute(Attribute attribute) { t.visitClassAttribute(attribute); }
        @Override public void visitNestMember(String nestMember) { t.visitNestMember(nestMember); }
        @Override public void visitPermittedSubclass(String permittedSubclass) { t.visitPermittedSubclass(permittedSubclass); }
        @Override public void visitInnerClass(String name, String outerName, String innerName, int access) { t.visitInnerClass(name, outerName, innerName, access); }
        @Override public RecordComponentVisitor visitRecordComponent(String name, String descriptor, String signature) { return new TraceRecordComponentVisitor(t.visitRecordComponent(name, descriptor, signature)); }
        @Override public FieldVisitor visitField(int access, String name, String descriptor, String signature, Object value) { return new TraceFieldVisitor(t.visitField(access, name, descriptor, signature, value)); }
        @Override public void visitEnd() { t.visitClassEnd(); }
        //@formatter:on
    }

    private static class LabelSettingTextifier extends Textifier {

        public LabelSettingTextifier() {
            super(Opcodes.ASM9);
        }

        @Override
        protected Textifier createTextifier() {
            return new LabelSettingTextifier();
        }

        public void setLabels(List<Label> labels) {
            labelNames = new HashMap<>();
            for (Label label : labels) {
                labelNames.put(label, "L" + labelNames.size());
            }
        }
    }
}
