package net.covers1624.coffeegrinder.bytecode.transform.transformers.statement;

import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import net.covers1624.coffeegrinder.bytecode.matching.LdcMatching;
import net.covers1624.coffeegrinder.bytecode.matching.LoadStoreMatching;
import net.covers1624.coffeegrinder.bytecode.transform.StatementTransformContext;
import net.covers1624.coffeegrinder.bytecode.transform.StatementTransformer;
import org.jetbrains.annotations.Nullable;

import java.util.LinkedList;
import java.util.List;

import static net.covers1624.quack.collection.FastStream.of;

/**
 * Created by covers1624 on 3/9/21.
 */
public class ArrayInitializers implements StatementTransformer {

    @Override
    public void transform(Instruction statement, StatementTransformContext ctx) {
        Store store = LoadStoreMatching.matchStoreLocal(statement);
        if (store == null || store.getVariable().getKind() != LocalVariable.VariableKind.STACK_SLOT) return;

        Instruction insn = store.getValue();
        if (!(insn instanceof NewArray newArray)) return;

        if (newArray.isInitializer || newArray.getIndices().size() != 1) return;

        LdcNumber ldc = LdcMatching.matchLdcInt(newArray.getIndices().first());
        if (ldc == null) return;
        int numElements = ldc.intValue();

        List<Store> elements = new LinkedList<>();
        Instruction pointer = store.getNextSiblingOrNull();
        for (int i = 0; i < numElements; i++) {
            if (pointer == null) return;

            Store element = getStoreElement(pointer, store.getVariable(), i);
            if (element == null) return;

            elements.add(element);
            pointer = pointer.getNextSiblingOrNull();
        }
        ctx.pushStep("Create array initializer");
        ctx.requestRerun();
        newArray.replaceWith(new NewArray(newArray.getType(), true, of(elements).map(Store::getValue)));
        elements.forEach(Instruction::remove);
        ctx.popStep();
    }

    @Nullable
    private Store getStoreElement(Instruction insn, LocalVariable arrayVar, int index) {
        if (!(insn instanceof Store store)) return null;
        if (!(store.getReference() instanceof ArrayElementReference elemRef)) return null;
        if (LoadStoreMatching.matchLoadLocal(elemRef.getArray(), arrayVar) == null) return null;
        if (LdcMatching.matchLdcInt(elemRef.getIndex(), index) == null) return null;

        return store;
    }
}
