package net.covers1624.coffeegrinder.type;

import net.covers1624.coffeegrinder.bytecode.AccessFlag;
import net.covers1624.coffeegrinder.util.EnumBitSet;
import net.covers1624.quack.collection.FastStream;
import org.objectweb.asm.Type;

import java.util.List;

import static net.covers1624.coffeegrinder.type.TypeSubstitutions.subst;
import static net.covers1624.quack.collection.FastStream.of;
import static net.covers1624.quack.util.SneakyUtils.unsafeCast;

/**
 * Created by covers1624 on 21/2/22.
 */
public class ParameterizedMethod extends Method {

    private final ClassType outer;
    private final Method prevDecl;
    private final List<TypeParameter> boundTypeParams;
    private final List<ReferenceType> typeArguments;
    private final AType boundReturnType;
    private final List<ReferenceType> boundExceptionTypes;

    private final List<Parameter> parameters;

    public ParameterizedMethod(ClassType outer, Method method, TypeSubstitutions.TypeParamMapper mapper, boolean supplyingArgs) {
        this.outer = outer;
        prevDecl = method;

        // type params
        assert !(method instanceof ParameterizedMethod) || supplyingArgs && ((ParameterizedMethod) method).getTypeArguments().isEmpty();
        List<TypeParameter> typeParams = method.getTypeParameters();
        if (typeParams.isEmpty()) {
            boundTypeParams = List.of();
            typeArguments = List.of();
        } else if (supplyingArgs) {
            boundTypeParams = List.of();
            typeArguments = FastStream.of(typeParams).map(mapper.substFunc()).toImmutableList();
        } else {
            typeArguments = List.of();
            class BoundTypeParameter extends TypeParameter {

                ReferenceType upper;

                public BoundTypeParameter(TypeParameter orig) {
                    super(orig.getName(), orig.getIndex(), ParameterizedMethod.this);
                    upper = orig.getUpperBound();
                }

                @Override
                public ReferenceType getUpperBound() {
                    return upper;
                }
            }

            List<BoundTypeParameter> boundParams = FastStream.of(typeParams).map(BoundTypeParameter::new).toImmutableList();

            TypeSubstitutions.TypeParamMapper outer_mapper = mapper;
            mapper = (typeParam) -> {
                if (typeParam.getOwner() == method) return boundParams.get(typeParam.getIndex());
                return outer_mapper.mapParam(typeParam);
            };

            for (BoundTypeParameter p : boundParams) {
                p.upper = subst(p.upper, mapper);
            }
            boundTypeParams = unsafeCast(boundParams);
        }

        boundReturnType = subst(method.getReturnType(), mapper);

        TypeSubstitutions.TypeParamMapper finalMapper = mapper;
        parameters = of(method.getParameters())
                .map(e -> new Parameter(
                        e.getName(),
                        this,
                        subst(e.getType(), finalMapper),
                        e.getRawType(),
                        e.getFlags()
                ))
                .toImmutableList();

        boundExceptionTypes = of(method.getExceptions())
                .map(e -> subst(e, finalMapper))
                .toImmutableList();
    }

    //@formatter:off
    @Override public Type getDescriptor() { return prevDecl.getDescriptor(); }
    @Override public Object getDefaultAnnotationValue() { throw new UnsupportedOperationException(); }
    @Override public ClassType getDeclaringClass() { return outer; }
    @Override public List<Parameter> getParameters() { return parameters; }
    @Override public AType getReturnType() { return boundReturnType; }
    @Override public boolean isConstructor() { return prevDecl.isConstructor(); }
    @Override public EnumBitSet<AccessFlag> getAccessFlags() { return prevDecl.getAccessFlags(); }
    @Override public List<ReferenceType> getExceptions() { return boundExceptionTypes; }
    @Override public Method getDeclaration() { return prevDecl.getDeclaration(); }
    @Override public Method asRaw() { return prevDecl.asRaw(); }
    @Override public String getName() { return prevDecl.getName(); }
    //@formatter:on

    public List<ReferenceType> getTypeArguments() {
        return typeArguments;
    }

    @Override
    public List<TypeParameter> getTypeParameters() {
        if (!typeArguments.isEmpty()) throw new UnsupportedOperationException();

        return boundTypeParams;
    }

    @Override
    public boolean hasTypeParameters() {
        return !boundTypeParams.isEmpty();
    }

    public boolean isFullyParameterized() {
        return getTypeArguments().size() == prevDecl.getTypeParameters().size();
    }
}
