package net.covers1624.coffeegrinder.bytecode.insns;

import net.covers1624.coffeegrinder.bytecode.*;
import net.covers1624.coffeegrinder.type.AType;
import net.covers1624.coffeegrinder.util.EnumBitSet;
import net.covers1624.quack.collection.FastStream;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;

import static net.covers1624.coffeegrinder.bytecode.InstructionFlag.END_POINT_UNREACHABLE;

/**
 * Created by covers1624 on 24/2/21.
 */
public final class BlockContainer extends Instruction {

    public final InstructionCollection<Block> blocks = new InstructionCollection<>(this);

    private final List<Leave> leaves = new LinkedList<>();

    @Nullable
    private Block entryPoint;

    public BlockContainer() {
    }

    @Override
    public AType getResultType() {
        throw new UnsupportedOperationException("BlockContainer does not have a result type.");
    }

    @Override
    protected EnumBitSet<InstructionFlag> computeFlags() {
        EnumBitSet<InstructionFlag> flags = InstructionFlag.NONE.copy();
        for (Block block : blocks) {
            flags.or(block.getFlags());
        }

        if (leaves.isEmpty()) {
            flags.set(END_POINT_UNREACHABLE);
        } else {
            flags.clear(END_POINT_UNREACHABLE);
        }
        return flags;
    }

    @Override
    public EnumBitSet<InstructionFlag> getDirectFlags() {
        return InstructionFlag.NONE;
    }

    @Override
    protected void onChildModified() {
        setEntryPoint((Block) getFirstChildOrNull());
    }

    @Override
    protected void onConnected() {
        super.onConnected();
        if (entryPoint != null) {
            entryPoint.setEntryPoint(true);
        }
    }

    @Override
    protected void onDisconnected() {
        super.onDisconnected();
        if (entryPoint != null) {
            entryPoint.setEntryPoint(false);
        }
    }

    @Override
    public <R, C> R accept(InsnVisitor<R, C> visitor, C ctx) {
        return visitor.visitBlockContainer(this, ctx);
    }

    public static BlockContainer findClosestContainer(Instruction insn) {
        while (!(insn instanceof BlockContainer)) {
            insn = insn.getParent();
        }
        return (BlockContainer) insn;
    }

    void addLeave(Leave leave) {
        leaves.add(leave);
        invalidateFlags();
    }

    void remLeave(Leave leave) {
        leaves.remove(leave);
        invalidateFlags();
    }

    //@formatter:off
    @Nullable public Block getEntryPointOrNull() { return entryPoint; }
    public Block getEntryPoint() { return Objects.requireNonNull(getEntryPointOrNull()); }
    public int getLeaveCount() { return leaves.size(); }
    public FastStream<Leave> getLeaves() { return FastStream.of(leaves); }
    //@formatter:on

    private void setEntryPoint(@Nullable Block entryPoint) {
        if (this.entryPoint != null && isConnected()) {
            this.entryPoint.setEntryPoint(false);
        }
        this.entryPoint = entryPoint;
        if (this.entryPoint != null && isConnected()) {
            this.entryPoint.setEntryPoint(true);
        }
    }
}
