package net.covers1624.coffeegrinder.bytecode.transform;

import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.InstructionFlag;
import net.covers1624.coffeegrinder.bytecode.SimpleInsnVisitor;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import net.covers1624.coffeegrinder.util.None;
import org.jetbrains.annotations.Nullable;

/**
 * Created by covers1624 on 30/11/22.
 */
public class ExitPointCleanup extends SimpleInsnVisitor<MethodTransformContext> implements MethodTransformer {

    @Override
    public void transform(MethodDecl function, MethodTransformContext ctx) {
        function.accept(this, ctx);
    }

    @Override
    public None visitBlock(Block block, MethodTransformContext ctx) {
        unwrapLeaveToContinueIn(block, ctx);
        return super.visitBlock(block, ctx);
    }

    @Override
    public None visitTryCatch(TryCatch tryCatch, MethodTransformContext ctx) {
        super.visitTryCatch(tryCatch, ctx);
        cleanupOrphanedExitAfterCatchFixups(tryCatch, ctx);
        return NONE;
    }

    private void unwrapLeaveToContinueIn(Block block, MethodTransformContext ctx) {
        Instruction insn = block.instructions.secondToLastOrDefault();
        if (!(insn instanceof BlockContainer container)) return;

        unwrapLeaveToContinue(container, ctx);
    }

    private void unwrapLeaveToContinue(BlockContainer container, MethodTransformContext ctx) {
        if (container.blocks.size() > 1) return;

        Block parent = (Block) container.getParent();
        Instruction exit = container.getNextSibling();
        Continue cont = matchContinueOrLeaveToContinue(exit);
        if (cont == null) return;

        ctx.pushStep("Unwrap container with break to continue");
        for (Leave e : container.getLeaves().toList()) {
            e.replaceWith(new Continue(cont.getLoop()).withOffsets(e));
        }

        exit.remove();

        Block b = (Block) container.getFirstChild();
        parent.instructions.addAll(b.instructions);
        container.remove();

        ctx.popStep();
    }

    @Nullable
    private Continue matchContinueOrLeaveToContinue(Instruction insn) {
        if (insn instanceof Continue cont) return cont;
        if (insn instanceof Leave leave) return followFlowToContinue(leave);
        return null;
    }

    private void cleanupOrphanedExitAfterCatchFixups(TryCatch tryCatch, MethodTransformContext ctx) {
        if (!(tryCatch.getParent() instanceof Block)) return;

        Instruction next = tryCatch.getNextSiblingOrNull();
        if (!tryCatch.hasFlag(InstructionFlag.END_POINT_UNREACHABLE) || next == null) return;
        assert next.hasFlag(InstructionFlag.END_POINT_UNREACHABLE);

        ctx.pushStep("Remove unreachable fallthrough");
        next.remove();
        ctx.popStep();
    }

    @Nullable
    private Continue followFlowToContinue(Leave leave) {
        if (!(leave.getTargetContainer().getParent() instanceof TryCatch.TryCatchHandler)) return null;

        Instruction next = leave.getTargetContainer().getParent().getParent().getNextSibling();
        if (next instanceof Continue cont) return cont;
        if (next instanceof Leave leave1) return followFlowToContinue(leave1);
        return null;
    }
}
