package net.covers1624.coffeegrinder.type;

import net.covers1624.quack.collection.FastStream;
import org.jetbrains.annotations.Nullable;
import org.objectweb.asm.Type;

import java.util.*;

import static net.covers1624.quack.collection.FastStream.*;

/**
 * Created by covers1624 on 1/9/21.
 */
public abstract class ReferenceType extends AType {

    @Nullable
    private List<Method> allMethods = null;

    public abstract ReferenceType getSuperType();

    // the actual type will be assignable to the upper bound
    // anything the upper bound is assignable to, the actual type will be assignable to
    public ReferenceType getUpperBound() {
        return this;
    }

    // the lower bound will be assignable to the actual type
    // anything assignable to the lower bound, is assignable to the actual type
    public ReferenceType getLowerBound() {
        return this;
    }

    public FastStream<? extends ReferenceType> getDirectSuperTypes() {
        return of(getSuperType());
    }

    @Nullable
    public Field resolveField(String name, Type desc) {
        return getDirectSuperTypes()
                .flatMap(e -> ofNullable(e.resolveField(name, desc))).firstOrDefault();
    }

    @Nullable
    public Method resolveMethod(String name, Type desc) {
        return getDirectSuperTypes()
                .flatMap(e -> ofNullable(e.resolveMethod(name, desc))).firstOrDefault();
    }

    public List<Method> getMethods() {
        return List.of();
    }

    public List<Method> getAllMethods() {
        if (allMethods == null) {
            allMethods = computeAllMethods();
        }
        return allMethods;
    }

    private List<Method> computeAllMethods() {
        LinkedHashSet<Method> methods = new LinkedHashSet<>();
        HashSet<String> signatures = new HashSet<>();
        for (Method m : getMethods()) {
            String desc = m.getDescriptor().toString();
            int lastBracket = desc.indexOf(')');
            String sig = m.getName() + desc.substring(0, lastBracket + 1);

            methods.add(m);
            signatures.add(sig);
        }

        for (Method m : getDirectSuperTypes().flatMap(ReferenceType::getAllMethods)) {
            String desc = m.getDescriptor().toString();
            int lastBracket = desc.indexOf(')');
            String sig = m.getName() + desc.substring(0, lastBracket + 1);

            if (!signatures.contains(sig))
                methods.add(m);
        }

        return List.copyOf(methods);
    }

    @Nullable
    public Method getFunctionalInterfaceMethod() {
        return getSuperType().getFunctionalInterfaceMethod();
    }

    @Override
    public boolean mentions(ReferenceType type) {
        return this == type;
    }
}
