package net.covers1624.coffeegrinder.bytecode.matching;

import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import net.covers1624.coffeegrinder.bytecode.insns.Invoke.InvokeKind;
import net.covers1624.coffeegrinder.type.ClassType;
import net.covers1624.coffeegrinder.type.Method;
import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.Nullable;
import org.objectweb.asm.Type;

/**
 * Created by covers1624 on 22/6/21.
 */
public class InvokeMatching {

    @Nullable
    public static Invoke matchInvoke(@Nullable Instruction insn, InvokeKind kind) {
        if (!(insn instanceof Invoke invoke)) return null;
        if (invoke.getKind() != kind) return null;

        return invoke;
    }

    @Nullable
    public static Invoke matchInvoke(@Nullable Instruction insn, InvokeKind kind, Method method) {
        Invoke invoke = matchInvoke(insn, kind, method.getName(), method.getDescriptor());
        if (invoke == null) return null;

        if (!invoke.getMethod().equals(method)) return null;

        return invoke;
    }

    @Nullable
    public static Invoke matchInvoke(@Nullable Instruction insn, InvokeKind kind, String name) {
        Invoke invoke = matchInvoke(insn, kind);
        if (invoke == null) return null;

        Method m = invoke.getMethod();
        if (!m.getName().equals(name)) return null;

        return invoke;
    }

    @Nullable
    public static Invoke matchInvoke(@Nullable Instruction insn, InvokeKind kind, String name, Type desc) {
        Invoke invoke = matchInvoke(insn, kind, name);
        if (invoke == null) return null;

        Method m = invoke.getMethod();
        if (!m.getDescriptor().equals(desc)) return null;

        return invoke;
    }

    @Nullable
    public static Invoke matchConstructorInvokeSpecial(@Nullable Instruction insn, ClassType decl) {
        Invoke invoke = matchInvoke(insn, InvokeKind.SPECIAL);
        if (invoke == null) return null;

        Method method = invoke.getMethod().getDeclaration();
        if (!method.isConstructor()) return null;
        if (!method.getDeclaringClass().equals(decl)) return null;
        return invoke;
    }

    @Nullable
    public static Invoke getSuperConstructorCall(MethodDecl functionInsn) {
        if (!functionInsn.getMethod().isConstructor()) return null;
        ClassType superClass = functionInsn.getMethod().getDeclaringClass().getSuperClass().getDeclaration();

        Instruction instruction = functionInsn.getBody().getEntryPoint().getFirstChild();
        while (matchInvoke(instruction, InvokeKind.SPECIAL, "<init>") == null) {
            // for valid code output, the only instructions allowed before the constructor call at this stage are synthetic field stores for
            // outer class field copies. However if inlining has failed for some reason, there could be other code before the constructor.
            // assert LoadStoreMatching.matchStoreField(instruction) != null;
            instruction = instruction.getNextSiblingOrNull();
            if (instruction == null) return null;
        }

        return matchConstructorInvokeSpecial(instruction, superClass);
    }

    @Nullable
    public static New matchNew(@Nullable Instruction insn, Method method) {
        if (!(insn instanceof New newInsn)) return null;
        if (!newInsn.getMethod().equals(method)) return null;

        return newInsn;
    }

    @Nullable
    public static New matchNew(@Nullable Instruction insn, ClassType type) {
        if (!(insn instanceof New newInsn)) return null;
        if (!newInsn.getResultType().equals(type)) return null;

        return newInsn;
    }

    @Nullable
    @Contract ("null,_->null")
    public static InvokeDynamic matchInvokeDynamic(@Nullable Instruction insn, ClassType boostrapClass) {
        if (!(insn instanceof InvokeDynamic indy)) return null;

        if (!indy.bootstrapHandle.getDeclaringClass().getDeclaration().equals(boostrapClass)) return null;

        return indy;
    }

    @Nullable
    @Contract ("null,_,_->null")
    public static InvokeDynamic matchInvokeDynamic(@Nullable Instruction insn, ClassType boostrapClass, String boostrapMethod) {
        InvokeDynamic indy = matchInvokeDynamic(insn, boostrapClass);
        if (indy == null) return null;

        if (!indy.bootstrapHandle.getName().equals(boostrapMethod)) return null;

        return indy;
    }
}
