package net.covers1624.coffeegrinder.bytecode.transform.transformers;

import net.covers1624.coffeegrinder.bytecode.AccessFlag;
import net.covers1624.coffeegrinder.bytecode.IndexedInstructionCollection;
import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import net.covers1624.coffeegrinder.bytecode.matching.InvokeMatching;
import net.covers1624.coffeegrinder.bytecode.transform.ClassTransformContext;
import net.covers1624.coffeegrinder.bytecode.transform.ClassTransformer;
import net.covers1624.coffeegrinder.type.TypeResolver;
import org.objectweb.asm.Type;

import java.util.List;

import static net.covers1624.coffeegrinder.bytecode.matching.LdcMatching.matchLdcInt;

/**
 * Created by covers1624 on 6/9/21.
 */
public class EnumClasses implements ClassTransformer {

    @Override
    public void transform(ClassDecl cInsn, ClassTransformContext ctx) {
        if (!cInsn.getClazz().isEnum()) return;
        // TODO store ClassType for Enum as instance field as per other transforms.
        if (cInsn.getClazz().getSuperClass().getDeclaration() != ctx.getTypeResolver().resolveClassDecl(TypeResolver.ENUM_TYPE)) return;

        List<FieldDecl> enumConstants = cInsn.getFieldMembers()
                .filter(e -> e.getField().getAccessFlags().get(AccessFlag.ENUM))
                .toList();
        ctx.pushStep("Remove implicit constructor params");
        for (int i = 0; i < enumConstants.size(); i++) {
            FieldDecl enumConstant = enumConstants.get(i);
            assert enumConstant.getValue() instanceof New;

            New newEnum = (New) enumConstant.getValue();
            assert newEnum.getTarget() == null;

            IndexedInstructionCollection<Instruction> args = newEnum.getArguments();
            assert args.size() >= 2;
            assert args.get(0) instanceof LdcString;
            assert matchLdcInt(args.get(1)) != null;

            assert enumConstant.getField().getName().equals(((LdcString) args.get(0)).getValue()) : "Enum constant name violation";
            assert i == ((LdcNumber) args.get(1)).intValue() : "Enum constant index violation";
            args.get(0).replaceWith(new Nop());
            args.get(1).replaceWith(new Nop());
        }

        Type arrayType = Type.getType("[" + cInsn.getClazz().getDescriptor());
        FieldDecl valuesField = cInsn.findField("$VALUES", arrayType);
        if (valuesField == null) {
            // Try the ECJ name of the field: https://github.com/eclipse-jdt/eclipse.jdt.core/blob/8f22f2778bc4fa09a0ecb12de521277a18db946e/org.eclipse.jdt.core.compiler.batch/src/org/eclipse/jdt/internal/compiler/lookup/TypeConstants.java#L554
            valuesField = cInsn.getField("ENUM$VALUES", arrayType);
        }
        valuesField.remove();
        ctx.popStep();

        // Nuke effectively synthetic methods.
        ctx.pushStep("Remove effectively synthetic methods");
        cInsn.getMethod("values", Type.getMethodType(arrayType)).remove();
        cInsn.getMethod("valueOf", Type.getMethodType(cInsn.getClazz().getDescriptor(), TypeResolver.STRING_TYPE)).remove();
        ctx.popStep();

        ctx.pushStep("Remove synthetic constructor parameters");
        cInsn.getMethodMembers()
                .filter(e -> e.getMethod().isConstructor())
                .forEach(m -> {
                    Invoke invoke = findConstructorChain(m);
                    invoke.getArguments().first().replaceWith(new Nop());
                    invoke.getArguments().get(1).replaceWith(new Nop());

                    ParameterVariable rem1 = m.parameters.get(0);
                    ParameterVariable rem2 = m.parameters.get(1);
                    rem1.makeImplicit();
                    rem2.makeImplicit();
                    assert rem1.getReferenceCount() == 0;
                    assert rem2.getReferenceCount() == 0;
                });
        ctx.popStep();
    }

    private Invoke findConstructorChain(MethodDecl function) {
        return (Invoke) function.getBody().getEntryPoint().instructions
                .filter(e -> InvokeMatching.matchInvoke(e, Invoke.InvokeKind.SPECIAL, "<init>") != null)
                .only();

    }

}
