package net.covers1624.coffeegrinder.bytecode.transform.transformers.generics;

import net.covers1624.coffeegrinder.bytecode.IndexedInstructionCollection;
import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import net.covers1624.coffeegrinder.type.*;
import net.covers1624.coffeegrinder.type.TypeSubstitutions.TypeMapper;
import net.covers1624.coffeegrinder.type.TypeSubstitutions.TypeSubstApplier;
import net.covers1624.coffeegrinder.util.Util;
import net.covers1624.quack.collection.ColUtils;
import net.covers1624.quack.collection.FastStream;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;

import static java.util.Objects.requireNonNull;
import static net.covers1624.coffeegrinder.type.TypeSystem.*;
import static net.covers1624.quack.collection.FastStream.of;
import static net.covers1624.quack.util.SneakyUtils.notPossible;
import static net.covers1624.quack.util.SneakyUtils.unsafeCast;

/**
 * Created by covers1624 on 20/6/22.
 */
public abstract class BoundSet {

    protected final Map<TypeParameter, InferenceVar> vars = new LinkedHashMap<>();
    protected final AType infVarRetType; // ret type of the expression, with inf vars substituted. Important to hold to preserve capture identity throughout the inference process

    protected final Map<InferenceVar, VarBounds<?>> bounds = new LinkedHashMap<>();
    protected final Map<AbstractInvoke, BoundSet> nestedVars = new LinkedHashMap<>();

    protected boolean hasRawArgs;

    @Nullable
    protected String failure = null;

    // just for distinguishing vars in nested trees
    @Nullable
    private BoundSet parent;
    private int nestedIndex;
    private int numNestedChildren;

    protected BoundSet(BoundSet other) {
        vars.putAll(other.vars);
        for (Map.Entry<InferenceVar, VarBounds<?>> entry : other.bounds.entrySet()) {
            bounds.put(entry.getKey(), copyVarBounds(entry.getValue()));
        }

        infVarRetType = other.infVarRetType;
        hasRawArgs = other.hasRawArgs;
        nestedVars.putAll(other.nestedVars);
        failure = other.failure;

        parent = other.parent;
        nestedIndex = other.nestedIndex;
        numNestedChildren = other.numNestedChildren;
    }

    public BoundSet(Iterable<TypeParameter> typeParameters, AType retType) {
        for (TypeParameter param : typeParameters) {
            InferenceVar v = new InferenceVar(param.getName() + "?", param);
            vars.put(param, v);
            bounds.put(v, newVarBounds());
        }

        // initial bounds from type param bounds
        for (TypeParameter param : typeParameters) {
            subtype(var(param), substInfVars(param.getUpperBound()));
        }
        for (TypeParameter param : typeParameters) {
            boundsFor(param).reverseUppers();
        }

        infVarRetType = retType instanceof ReferenceType ? substInfVars((ReferenceType) retType) : retType;
    }

    protected VarBounds<?> newVarBounds() {
        return new SimpleVarBounds(this);
    }

    protected VarBounds<?> copyVarBounds(VarBounds<?> other) {
        throw new UnsupportedOperationException("This bound set cannot be copied.");
    }

    // region Constraints (Reduction)
    public void constrainThrown(ReferenceType exType) {
        if (exType instanceof TypeParameter) {
            boundsFor((TypeParameter) exType).thrown = true;
        }
    }

    public void constrainAssignable(Instruction insn, AType type) {
        if (!(type instanceof ReferenceType)) return;

        assignable(insn, substInfVars((ReferenceType) type));
    }

    public void constrainReturnAssignable(ReferenceType polyResultType) {
        // shouldn't subst inf vars on polyResultType because the inference vars may be for a class, and the ret type may actually be one of those type params
        // eg, new ArrayList<>(elements); where elements is E[] inside ArrayList<E>
        assignable(infVarRetType, polyResultType);
    }
    // endregion

    // region Resolution

    protected InferenceVarMapper solve() {
        InferenceVarMapper mapper;
        Map<InferenceVar, ReferenceType> solution = new HashMap<>();
        mapper = p -> solution.getOrDefault(p, p);
        for (List<InferenceVar> vars : new TarjanDepthFirstIterator<>(bounds.keySet(), v -> boundsFor(v).getDeps())) {
            solution.putAll(solveVars(vars, mapper));
        }

        assert bounds.size() == solution.size();
        return mapper;
    }

    protected Map<InferenceVar, ReferenceType> solveVars(List<InferenceVar> vars, InferenceVarMapper solved) {
        Map<InferenceVar, ReferenceType> solution = solvePhases(vars, solved);
        if (solution.size() == vars.size() && check(solution, solved)) {
            return solution;
        }

        // try again with fresh type vars
        return solveFreshUpper(vars, solved);
    }

    protected final Map<InferenceVar, ReferenceType> solvePhases(List<InferenceVar> vars, InferenceVarMapper solved) {
        Map<InferenceVar, ReferenceType> solution = new HashMap<>();
        InferenceVarMapper partialSolution = p -> solution.getOrDefault(p, solved.mapType(p));
        for (ResolutionPhase phase : ResolutionPhase.values()) {
            Map<InferenceVar, ReferenceType> solvedThisPhase = new HashMap<>();
            for (InferenceVar var : vars) {
                if (solution.containsKey(var)) continue;

                ReferenceType t = solvePhase(phase, boundsFor(var), partialSolution);
                if (t != null) {
                    assert isProper(t);
                    solvedThisPhase.put(var, t);
                }
            }
            solution.putAll(solvedThisPhase);
        }
        return solution;
    }

    @Nullable
    protected ReferenceType solvePhase(ResolutionPhase phase, VarBounds<?> bounds, InferenceVarMapper solved) {
        return phase.solve(bounds, solved);
    }

    private boolean check(Map<InferenceVar, ReferenceType> solution, InferenceVarMapper solved) {
        InferenceVarMapper combined = v -> solution.getOrDefault(v, solved.mapType(v));
        return ColUtils.allMatch(solution.keySet(), v -> check(v, combined.substFunc()));
    }

    private boolean check(InferenceVar var, TypeSubstApplier subst) {
        ReferenceType varResult = subst.apply(var);
        VarBounds<?> bounds = boundsFor(var);
        return bounds.equalTypes(subst).allMatch(e -> subst.apply(e).equals(varResult))
               && bounds.lowerTypes(subst).allMatch(l -> TypeSystem.isAssignableTo(subst.apply(l), varResult))
               && bounds.upperTypes(subst).allMatch(u -> TypeSystem.isAssignableTo(varResult, subst.apply(u)));
    }

    protected enum ResolutionPhase {
        EQUALS {
            @Nullable
            @Override
            public ReferenceType solve(VarBounds<?> bounds, InferenceVarMapper partialSolution) {
                return bounds.equalTypes(partialSolution.substFunc())
                        .filter(BoundSet::isProper)
                        .firstOrDefault();
            }
        },
        LOWER {
            @Nullable
            @Override
            public ReferenceType solve(VarBounds<?> bounds, InferenceVarMapper partialSolution) {
                ReferenceType[] lower = bounds.lowerTypes(partialSolution.substFunc())
                        .filter(BoundSet::isProper)
                        .toArray(new ReferenceType[0]);

                return lower.length > 0 ? TypeSystem.lub(lower) : null;
            }
        },
        UPPER { // TODO: throws,captured

            @Nullable
            @Override
            public ReferenceType solve(VarBounds<?> bounds, InferenceVarMapper partialSolution) {
                List<ReferenceType> uppers = bounds.upperTypes(partialSolution.substFunc())
                        .filter(BoundSet::isProper)
                        .toList();

                if (!uppers.isEmpty()) {
                    Util.reverse(uppers); // required to get the right behavior with glbJavac
                    checkSingleMostDerivedClassType(uppers);

                    return TypeSystem.glbJavac(uppers);
                }

                return null;
            }
        };

        private static void checkSingleMostDerivedClassType(List<ReferenceType> types) {
            for (int i = 0; i < types.size(); i++) {
                ReferenceType a = types.get(i);
                for (int j = i; j < types.size(); j++) {
                    ReferenceType b = types.get(j);

                    if (!isInterface(a) && !isInterface(b) && !isAssignableTo(a, b) && !isAssignableTo(b, a)) { throw new ResolveFailedException("Cannot find glb of distinct types " + a + " and " + b); }
                }
            }
        }

        @Nullable
        public abstract ReferenceType solve(VarBounds<?> bounds, InferenceVarMapper partialSolution);
    }

    private Map<InferenceVar, ReferenceType> solveFreshUpper(List<InferenceVar> vars, InferenceVarMapper solved) {
        abstract class FreshTypeVar extends TypeVariable {

            protected final InferenceVar var;
            protected ReferenceType upper;

            protected FreshTypeVar(InferenceVar var) {
                this.var = var;
                upper = TypeSystem.objectType(var);
            }

            @Override
            public ReferenceType getUpperBound() {
                return upper;
            }

            abstract void update(InferenceVarMapper freshVars);

            @Override
            public String toString() {
                return getName();
            }

            @Override
            public String getName() {
                return "ft:" + var.getName();
            }

            @Override
            public boolean equals(Object o) {
                if (this == o) return true;
                if (!(o instanceof FreshTypeVar)) return false;
                FreshTypeVar other = (FreshTypeVar) o;

                return other.var.equals(var);
            }

            @Override
            public int hashCode() {
                return var.hashCode();
            }
        }

        Map<InferenceVar, ReferenceType> solution = of(vars).toImmutableMap(
                Function.identity(),
                var -> {
                    List<ReferenceType> uppers = boundsFor(var)
                            .upperTypes(solved.substFunc())
                            .toList();
                    Util.reverse(uppers); // required to get the right behavior with glbJavac

                    if (ColUtils.allMatch(uppers, BoundSet::isProper)) {
                        return TypeSystem.glbJavac(uppers);
                    }

                    return new FreshTypeVar(var) {
                        @Override
                        void update(InferenceVarMapper freshVars) {
                            upper = TypeSystem.glbJavac(of(uppers).map(freshVars.substFunc()));
                        }
                    };
                }
        );

        InferenceVarMapper mapFresh = t -> solution.getOrDefault(t, t);
        for (ReferenceType t : solution.values()) {
            if (t instanceof FreshTypeVar) {
                ((FreshTypeVar) t).update(mapFresh);
            }
        }

        if (check(solution, solved)) {
            return solution;
        }

        throw new ResolveFailedException("Fresh type vars failed check");
    }
    // endregion

    // region Helpers / Interactions
    protected void addNestedVars(AbstractInvoke expr, BoundSet other) {
        assert other.parent == null;

        for (Map.Entry<InferenceVar, VarBounds<?>> entry : other.bounds.entrySet()) {
            bounds.put(entry.getKey(), entry.getValue().setRoot(this));
        }
        nestedVars.put(expr, other);
        other.parent = this;
        other.nestedIndex = numNestedChildren++;

        if (other.failure != null) {
            fail(other.failure);
        }
    }

    protected VarBounds<?> boundsFor(InferenceVar var) {
        return bounds.get(var);
    }

    protected VarBounds<?> boundsFor(TypeParameter param) {
        return boundsFor(var(param));
    }

    private InferenceVar var(TypeParameter param) {
        return requireNonNull(vars.get(param));
    }

    private ReferenceType subst(TypeParameter param) {
        InferenceVar var = vars.get(param);
        return var != null ? var : param;
    }

    private ReferenceType substInfVars(ReferenceType t) {
        return TypeSubstitutions.subst(t, (TypeSubstitutions.TypeParamMapper) this::subst);
    }

    public static boolean isPoly(Method method) {
        return method.isConstructor() ? method.getDeclaringClass().getDeclaration().hasTypeParameters() :
                method.hasTypeParameters() && mentionsTypeParam(method.getReturnType(), method);
    }

    @NotNull
    private static ClassType retTypeForConstructor(ClassType c) {
        if (TypeSystem.isFullyDefined(c)) return c;

        // except if the class is generic and still requires inference
        ClassType decl = c.getDeclaration();
        assert decl.hasTypeParameters();

        // If the outer is generic, it will be fully parameterized by now
        ParameterizedClass pOuter = c instanceof ParameterizedClass ? ((ParameterizedClass) c).getOuter() : null;
        return new ParameterizedClass(pOuter, decl, unsafeCast(decl.getTypeParameters()));
    }

    protected static boolean isProper(AType t) {
        return switch (t) {
            case InferenceVar inferenceVar -> false;
            case ParameterizedClass parameterizedClass -> of(parameterizedClass.getTypeArguments()).allMatch(BoundSet::isProper);
            case ArrayType arrayType -> isProper(arrayType.getElementType());
            case WildcardType wildcardType -> isProper(wildcardType.getUpperBound()) && isProper(wildcardType.getLowerBound());
            case IntersectionType intersectionType -> intersectionType.getDirectSuperTypes().allMatch(BoundSet::isProper);
            default -> true;
        };
    }

    public static Iterable<TypeParameter> getInferrableTypeParams(Method method) {
        Iterable<TypeParameter> variables = method.getTypeParameters();
        if (method.isConstructor()) {
            variables = FastStream.concat(method.getDeclaringClass().getDeclaration().getTypeParameters(), variables);
        }

        return variables;
    }

    public static boolean mentionsTypeParamFromClassOrOuter(AType t, ClassType clazz) {
        if (mentionsTypeParam(t, clazz)) return true;
        if (!TypeSystem.needsOuterParameterization(clazz)) return false;

        return mentionsTypeParamFromClassOrOuter(t, clazz.getEnclosingClass().orElseThrow(notPossible()));
    }

    public static boolean mentionsInferrableTypeParam(AType t, Method m) {
        return mentionsTypeParam(t, m) || m.isConstructor() && mentionsTypeParam(t, m.getDeclaringClass());
    }

    public static boolean mentionsTypeParam(AType t, ITypeParameterizedMember owner) {
        return t instanceof ReferenceType && mentionsTypeParam((ReferenceType) t, owner, e -> false);
    }

    public static boolean mentionsTypeParam(ReferenceType t, ITypeParameterizedMember owner, Predicate<TypeParameter> seen) {
        return switch (t) {
            case TypeParameter type ->
                    !seen.test(type)
                    && (type.getOwner() == owner || mentionsTypeParam(t.getSuperType(), owner, p -> p == t || seen.test(p)));
            case ParameterizedClass clazz -> of(clazz.getTypeArguments()).anyMatch(e -> mentionsTypeParam(e, owner, seen));
            case ArrayType type when type.getElementType() instanceof ReferenceType -> mentionsTypeParam((ReferenceType) type.getElementType(), owner, seen);
            case WildcardType type -> mentionsTypeParam(t.getUpperBound(), owner, seen) || mentionsTypeParam(t.getLowerBound(), owner, seen);
            default -> false;
        };
    }

    public static AType getHierarchyCompatibleType(AType targetType, AType inputType) {
        if (targetType instanceof ReferenceType && inputType instanceof ReferenceType) return getHierarchyCompatibleType((ReferenceType) targetType, (ReferenceType) inputType);
        return targetType;
    }

    public static ReferenceType getHierarchyCompatibleType(ReferenceType targetType, ReferenceType compatibleType) {
        if (targetType instanceof ClassType) return getHierarchyCompatibleType((ClassType) targetType, compatibleType);
        if (targetType instanceof ArrayType && compatibleType instanceof ArrayType) {
            return ((ArrayType) compatibleType).withElementType(
                    getHierarchyCompatibleType(((ArrayType) targetType).getElementType(), ((ArrayType) compatibleType).getElementType()));
        }
        return targetType;
    }

    // eg, compatibleType is Collection<String> and targetType is List<_>
    // This function makes a 'best attempt', and does not fail if incompatible types are detected
    // Type parameters with no mapping are replaced with unbounded wildcards
    // Assertion fails if different mappings are found for the same type parameter
    public static ClassType getHierarchyCompatibleType(ClassType targetType, ReferenceType compatibleType) {
        if (targetType.equals(compatibleType)) { return targetType; }

        // TODO: determine if this only does something when the types are castable, or if it's meant to work on disjoint types (eg ArrayList and ImmutableList)

        targetType = targetType.getDeclaration();
        if (!isGeneric(targetType)) return targetType;

        ClassType p = TypeSystem.makeThisType(targetType);
        // TODO, probably needs to go through outer args as well
        Map<TypeParameter, ReferenceType> mappings = new HashMap<>();

        for (ReferenceType t : TypeSystem.getCommonDeclaredHierarchy(p, compatibleType)) {
            if (!(t instanceof ClassType)) continue;

            ClassType c = (ClassType) t;
            if (!TypeSystem.isGeneric(c)) continue;

            ClassType c1 = TypeSystem.findParameterization(c, p);
            ClassType c2 = TypeSystem.findParameterization(c, compatibleType);
            if (c1 instanceof RawClass || c2 instanceof RawClass) continue;

            mapTypes(mappings, c1, c2);
        }

        return TypeSubstitutions.subst(p, (TypeSubstitutions.TypeParamMapper)
                param -> mappings.getOrDefault(param, WildcardType.createExtends(TypeSystem.objectType(p))));
    }

    private static final TypeMapper MAKE_REPRESENTABLE = type -> {
        // todo, also remove intersections
        if (type instanceof CapturedTypeVar) {
            return ((CapturedTypeVar) type).wildcard;
        }
        return type;
    };

    public static ReferenceType makeRepresentable(ReferenceType t) {
        return TypeSubstitutions.subst(t, MAKE_REPRESENTABLE);
    }
    // endregion

    // region Assignability Constraint to Bounds
    protected void assignable(Instruction expr, ReferenceType t) {
        if (expr instanceof MethodDecl decl) {
            assignable(decl, t);
            return;
        }

        if (expr instanceof MethodReference mref) {
            assignable(mref, t);
            return;
        }

        ReferenceType resultType = (ReferenceType) expr.getResultType();
        BoundSet polyBounds = null;
        if (expr instanceof AbstractInvoke invoke) {
            if (isPoly(invoke.getMethod())) {
                polyBounds = monomorphicBounds(invoke, this::makeNestedBoundSet);
                resultType = (ReferenceType) polyBounds.infVarRetType;
                addNestedVars(invoke, polyBounds);
            }
        }
        if (resultType == NullConstantType.INSTANCE) return;

        assignable(expr, resultType, t, polyBounds);
    }

    protected void assignable(Instruction expr, ReferenceType resultType, ReferenceType t, @Nullable BoundSet polyBounds) {
        if (resultType instanceof RawClass) {
            hasRawArgs = true;
        }
        assignable(resultType, t);
    }

    protected abstract BoundSet makeNestedBoundSet(Iterable<TypeParameter> vars, AType retType);

    protected void assignable(MethodDecl lambda, ReferenceType t) {
        if (t instanceof InferenceVar) {
            return; // TODO, post resolve check for assignability? Some other form of constraint? Use the InvokeDynamic target type as a bound instead?
        }

        Method fMethod = requireNonNull(t.getFunctionalInterfaceMethod());
        List<Parameter> fParams = fMethod.getParameters();

        List<AType> params = lambda.parameters.filterNot(ParameterVariable::isImplicit).map(LocalVariable::getType).toList();

        assert fParams.size() == params.size();
        lambdaParamsCanReceiveFunctionalInterfaceMethodType(params, fParams);

        if (fMethod.getReturnType() instanceof ReferenceType retType) {
            retTypeAssignable(lambda, upper(retType));
        }
    }

    // Note that this is not a part of standard inference.
    // The default implementation here is common and helpful to our use cases, but should be disabled when trying to match javac
    protected void lambdaParamsCanReceiveFunctionalInterfaceMethodType(List<AType> params, List<Parameter> fParams) {
        for (int i = 0; i < params.size(); i++) {
            assignableStripWildcards(fParams.get(i).getType(), params.get(i));
        }
    }

    protected void retTypeAssignable(MethodDecl lambda, ReferenceType retType) {
        for (Return ret : lambda.getReturns()) {
            assignable(ret.getValue(), retType);
        }
    }

    protected void assignable(MethodReference mref, ReferenceType t) {
        if (t instanceof InferenceVar) {
            return;
        }

        Method fMethod = requireNonNull(t.getFunctionalInterfaceMethod());
        assert fMethod.getDeclaration() == mref.getResultType().getDeclaration().getFunctionalInterfaceMethod().getDeclaration();

        if (fMethod.hasTypeParameters()) {
            // todo, anything?
            return;
        }

        List<Parameter> fParams = fMethod.getParameters();

        List<Method> implicitCandidates = getCandidates(mref).toLinkedList();
        boolean isExact = implicitCandidates.size() == 1;

        Method method = mref.getMethod();
        boolean instanceTypeIsFirstParam = !method.isStatic() && !method.isConstructor() && mref.getTarget() instanceof Nop;

        if (!TypeSystem.isFullyDefined(method) || (instanceTypeIsFirstParam || method.isConstructor()) && !TypeSystem.isFullyDefined(method.getDeclaringClass())) {
            // todo, create a delayed assignability check on the resolution
            //   generate some inference vars for the method, and add them as constraints (when considering explicit args)
            return;
        }

        if (isExact) {
            List<AType> params = FastStream.of(method.getParameters()).map(Parameter::getType).toList();
            if (instanceTypeIsFirstParam) {
                params.addFirst(method.getDeclaringClass());
            }

            assert fParams.size() == params.size();
            for (int i = 0; i < params.size(); i++) {
                assignableStripWildcards(fParams.get(i).getType(), params.get(i));
            }
        } else {
            //if (true)
            //    throw new NotImplementedException("Inexact(overloaded) method-ref inference");
            // postResolveChecks.add(compile-time-declaration assignable to resolved(t), resolutionStrategy...)

            // select a resolution strategy for if this mref is no-longer assignable after
            if (!(mref.getTarget() instanceof Nop) && getCandidates(method.getDeclaringClass(), method.getName()).count() == 1) {
                // resolution strategy = via widening cast
            } else {
                // try explicit type args
                // try raw-casting target
            }
        }

        if (fMethod.getReturnType() != PrimitiveType.VOID) {
            // TODO generic class constructors? is makeThisType correct?
            //      is that like nested inference for method references?
            AType retType = method.isConstructor() ? TypeSystem.makeThisType(method.getDeclaringClass()) : method.getReturnType();
            assignableStripWildcards(TypeSystem.capture(retType), fMethod.getReturnType());
        }
    }

    private static FastStream<Method> getCandidates(ReferenceType onType, String name) {
        // return all distinct overloads of name
        return FastStream.of(onType.getAllMethods()).filter(e -> e.getName().equals(name));
    }

    private static FastStream<Method> getCandidates(MethodReference mref) {
        return getCandidates(!(mref.getTarget() instanceof Nop) ? (ReferenceType) mref.getTarget().getResultType() : mref.getMethod().getDeclaringClass(), mref.getMethod().getName());
    }

    protected void assignableStripWildcards(AType s, AType t) {
        assignable(lower(s), upper(t));
    }

    public static AType upper(AType t) {
        return t instanceof ReferenceType ? upper((ReferenceType) t) : t;
    }

    private static ReferenceType upper(ReferenceType t) {
        if (t instanceof WildcardType) {
            assert !((WildcardType) t).isSuper();
            t = t.getUpperBound();
        }
        return t;
    }

    public static AType lower(AType t) {
        return t instanceof ReferenceType ? lower((ReferenceType) t) : t;
    }

    public static ReferenceType lower(ReferenceType t) {
        if (t instanceof WildcardType) {
            assert ((WildcardType) t).isSuper();
            t = t.getLowerBound();
        }
        return t;
    }

    private void assignable(AType s, AType t) {
        if (t instanceof PrimitiveType && s instanceof PrimitiveType) {
            assert isAssignableTo(s, t);
            return;
        }

        if (s instanceof PrimitiveType) {
            s = TypeSystem.box(TypeSystem.resolver((ReferenceType) t), (PrimitiveType) s);
        }

        if (t instanceof PrimitiveType) {
            t = TypeSystem.box(TypeSystem.resolver((ReferenceType) s), (PrimitiveType) t);
        }

        assignable((ReferenceType) s, (ReferenceType) t);
    }

    protected void assignable(ReferenceType s, ReferenceType t) {
        subtype(s, t);
    }

    public static <B extends BoundSet> B monomorphicBounds(AbstractInvoke invoke, BiFunction<Iterable<TypeParameter>, AType, B> factory) {
        Method method = invoke.getMethod();
        B b = newBoundSet(invoke, factory);
        // add throws bounds from type params present in method throws decl
        for (ReferenceType exception : method.getExceptions()) {
            b.constrainThrown(exception);
        }

        // parameter expression assignability reduction
        IndexedInstructionCollection<Instruction> args = invoke.getArguments();
        List<Parameter> params = method.getParameters();
        for (int i = 0; i < params.size(); i++) {
            if (args.get(i) instanceof Nop) continue;

            AType type = params.get(i).getType();
            if (!BoundSet.mentionsInferrableTypeParam(type, method)) continue;
            b.constrainAssignable(args.get(i), type);
        }

        return b;
    }

    public static <B extends BoundSet> B newBoundSet(AbstractInvoke invoke, BiFunction<Iterable<TypeParameter>, AType, B> factory) {
        Method method = invoke.getMethod();
        Iterable<TypeParameter> variables = BoundSet.getInferrableTypeParams(method);
        AType retType = invoke instanceof New ? BoundSet.retTypeForConstructor(method.getDeclaringClass()) : invoke.getResultType();
        return factory.apply(variables, retType);
    }
    // endregion

    // region Apply Bounds / Check / Incorporate
    protected void subtype(ReferenceType s, ReferenceType t) {
        if (t == NullConstantType.INSTANCE) {
            fail("Upper bound of null (cannot have a subtype of a type with no lower bound)");
            return;
        }

        if (t instanceof IntersectionType) {
            for (ReferenceType c : t.getDirectSuperTypes()) {
                subtype(s, c);
            }
            return;
        }

        if (s instanceof InferenceVar) {
            boundsFor((InferenceVar) s).addUpper(t);
        }
        if (t instanceof InferenceVar) {
            boundsFor((InferenceVar) t).addLower(s);
        }
        if (s instanceof InferenceVar || t instanceof InferenceVar) { return; }

        if (s instanceof CapturedTypeVar) {
            subtype(s.getUpperBound(), t);
            return;
        }

        if (t instanceof ParameterizedClass) {
            argsContainedBy(s, (ParameterizedClass) t);
            return;
        }
        if (t instanceof ArrayType) {
            while (s.getUpperBound() != s) {
                s = s.getUpperBound();
            }
            if (t.equals(s)) return;

            assert s instanceof ArrayType;
            subtype((ReferenceType) ((ArrayType) s).getElementType(), (ReferenceType) ((ArrayType) t).getElementType());
            return;
        }

        if (!TypeSystem.isAssignableTo(s, t)) {
            failProperSubtype(s, t);
        }
    }

    protected void failProperSubtype(ReferenceType s, ReferenceType t) {
        fail("Proper types incompatible: " + s + " is not a subtype of " + t);
    }

    private void argsContainedBy(ReferenceType s, ParameterizedClass t) {
        ClassType tOnS = TypeSystem.findParameterizationOrNull(t.getDeclaration(), s);
        if (tOnS == null) {
            fail(s + " not a subtype of " + t);
            return;
        }
        if (tOnS instanceof RawClass) {
            return;
        }

        ParameterizedClass p = (ParameterizedClass) tOnS;
        if (t.getOuter() != null) {
            assert p.getOuter() != null;
            argsContainedBy(p.getOuter(), t.getOuter());
        }

        List<ReferenceType> tArgs = t.getTypeArguments();
        List<ReferenceType> sArgs = p.getTypeArguments();
        for (int i = 0; i < tArgs.size(); i++) {
            ReferenceType tArg = tArgs.get(i);
            ReferenceType sArg = sArgs.get(i);
            containedBy(sArg, tArg);
        }
    }

    protected void containedBy(ReferenceType s, ReferenceType t) {
        if (!(t instanceof WildcardType)) {
            equal(s, t);
            return;
        }

        if (((WildcardType) t).isSuper()) {
            subtype(t.getLowerBound(), s.getLowerBound());
        }

        // at this point, t = ? extends ...
        if (s instanceof WildcardType) {
            s = s.getUpperBound();
        }

        subtype(s, t.getUpperBound());
    }

    private void equal(WildcardType s, WildcardType t) {
        if (s.isSuper() == t.isSuper()) {
            equal(s.getUpperBound(), t.getUpperBound());
            equal(s.getLowerBound(), t.getLowerBound());
            return;
        }
        fail(s + " not equal to " + t);
    }

    private void equal(ReferenceType s, ReferenceType t) {
        if (s.equals(t)) return;

        if (s instanceof WildcardType && t instanceof WildcardType) {
            equal((WildcardType) s, (WildcardType) t);
            return;
        }

        if (s instanceof WildcardType || t instanceof WildcardType) {
            // TODO, is this important? When? Should we check that the other type is within the bound of the wildcard?
            //   inference vars can't be set 'equal' to a wildcard
            //   maybe instead, compatibleSuperTypes should use a 'compatibility' check on the parameterized class args, instead of equal
            //fail(s + " not equal to " + t);
            return;
        }

        if (t instanceof InferenceVar) {
            boundsFor((InferenceVar) t).addEqual(s);
        }
        if (s instanceof InferenceVar) {
            boundsFor((InferenceVar) s).addEqual(t);
        }

        if (s instanceof InferenceVar || t instanceof InferenceVar) { return; }

        if (t instanceof ParameterizedClass tParam) {
            if (!(s instanceof ParameterizedClass sParam)) {
                fail(s + " cannot be equal to " + t);
                return;
            }

            if (tParam.getDeclaration() != sParam.getDeclaration()) {
                fail(s + " cannot be equal to " + t);
                return;
            }

            List<ReferenceType> tArgs = tParam.getTypeArguments();
            List<ReferenceType> sArgs = sParam.getTypeArguments();
            for (int i = 0; i < tArgs.size(); i++) {
                equal(sArgs.get(i), tArgs.get(i));
            }
            return;
        }

        if (t instanceof ArrayType) {
            if (!(s instanceof ArrayType)) {
                fail(s + " cannot be equal to " + t);
                return;
            }
            equal((ReferenceType) ((ArrayType) s).getElementType(), (ReferenceType) ((ArrayType) t).getElementType());
            return;
        }

        failProperTypesNotEqual(s, t);
    }

    protected void failProperTypesNotEqual(ReferenceType s, ReferenceType t) {
        fail(s + " not equal to " + t);
    }

    protected void fail(String reason) {
        if (failure == null) {
            failure = reason;
        }
    }

    private void compatibleSupertypes(ReferenceType t1, ReferenceType t2) {
        // Performance, nothing to do here.
        if (isObject(t1)) return;
        if (t1 instanceof InferenceVar || t2 instanceof InferenceVar) return; // can't check hierarchy of inference vars. Upper bounds are cross-copied anyway

        for (ReferenceType t : TypeSystem.getCommonDeclaredHierarchy(t1, t2)) {
            if (!(t instanceof ClassType c)) continue;

            if (!TypeSystem.isGeneric(c)) continue;

            ClassType c1 = TypeSystem.findParameterization(c, t1);
            ClassType c2 = TypeSystem.findParameterization(c, t2);
            if (c1 instanceof RawClass || c2 instanceof RawClass) continue;
            equal(c1, c2);
        }

        // TODO: check that there is only one erased class type
    }
    // endregion

    protected static abstract class VarBounds<T> {

        protected final List<T> equalBounds = new ArrayList<>(6);
        protected final List<T> lowerBounds = new ArrayList<>(6);
        protected final List<T> upperBounds = new ArrayList<>(6);
        public boolean thrown;
        private BoundSet root;

        public VarBounds(BoundSet root) {
            this.root = root;
        }

        protected VarBounds(BoundSet root, VarBounds<T> other) {
            this.root = root;
            equalBounds.addAll(other.equalBounds);
            lowerBounds.addAll(other.lowerBounds);
            upperBounds.addAll(other.upperBounds);
            thrown = other.thrown;
        }

        protected VarBounds<T> setRoot(BoundSet root) {
            this.root = root;
            return this;
        }

        protected BoundSet getRoot() {
            return root;
        }

        protected abstract T makeBound(ReferenceType t);

        protected abstract ReferenceType getType(T t);

        protected ReferenceType getResolvedType(T t, TypeSubstApplier subst) {
            return subst.apply(getType(t));
        }

        protected boolean dependsOn(T t, InferenceVar v) {
            return getType(t).mentions(v);
        }

        protected void boundAdded(T b) { }

        protected boolean isEncompassedBy(T b1, T b2) {
            return b1.equals(b2);
        }

        protected <T1, T2> void incorporate(T1 b1, T2 b2, BiConsumer<T1, T2> action) {
            action.accept(b1, b2);
        }

        public final void addEqual(ReferenceType t) {
            T b = makeBound(t);
            if (!isEncompassedBy(b, equalBounds)) {
                addEqualBound(b);
            }
        }

        public final void addUpper(ReferenceType t) {
            T b = makeBound(t);
            if (!isEncompassedBy(b, upperBounds)) {
                addUpperBound(b);
            }
        }

        public final void addLower(ReferenceType t) {
            T b = makeBound(t);
            if (!isEncompassedBy(b, lowerBounds)) {
                addLowerBound(b);
            }
        }

        protected boolean isEncompassedBy(T b, List<T> existingBounds) {
            return ColUtils.anyMatch(existingBounds, b2 -> isEncompassedBy(b, b2));
        }

        protected void equal(T t1, T t2) {
            root.equal(getType(t1), getType(t2));
        }

        protected void subtype(T t1, T t2) {
            root.subtype(getType(t1), getType(t2));
        }

        protected void compatibleSupertypes(T t1, T t2) {
            root.compatibleSupertypes(getType(t1), getType(t2));
        }

        protected void addEqualBound(T b) {
            add(equalBounds, b, (t, equal) -> equal(t, equal));
            incorporateList(b, upperBounds, (t, upper) -> subtype(t, upper));
            incorporateList(b, lowerBounds, (t, lower) -> subtype(lower, t));
        }

        protected void addUpperBound(T b) {
            add(upperBounds, b, (t, upper) -> compatibleSupertypes(upper, t));
            incorporateList(b, equalBounds, (t, equal) -> subtype(equal, t));
            incorporateList(b, lowerBounds, (t, lower) -> subtype(lower, t));
        }

        protected void addLowerBound(T b) {
            add(lowerBounds, b, null);
            incorporateList(b, equalBounds, (t, equal) -> subtype(t, equal));
            incorporateList(b, upperBounds, (t, upper) -> subtype(t, upper));
        }

        protected void add(List<T> bounds, T bound, @Nullable BiConsumer<T, T> func) {
            bounds.add(bound);
            boundAdded(bound);

            if (func != null) {
                incorporateList(bound, bounds, bounds.size() - 1, func);
            }
        }

        protected <T1, T2> void incorporateList(T1 bound, List<T2> bounds, BiConsumer<T1, T2> func) {
            incorporateList(bound, bounds, bounds.size(), func);
        }

        protected <T1, T2> void incorporateList(T1 bound, List<T2> bounds, int count, BiConsumer<T1, T2> func) {
            for (int i = count - 1; i >= 0; i--) {
                incorporate(bound, bounds.get(i), func);
            }
        }

        // todo, visitor that collects from the bounds themselves?
        public Iterable<InferenceVar> getDeps() {
            return FastStream.of(root.bounds.keySet()).filter(this::dependsOn);
        }

        private boolean dependsOn(InferenceVar v2) {
            return activeBounds(equalBounds).anyMatch(t -> dependsOn(t, v2)) ||
                   activeBounds(upperBounds).anyMatch(t -> dependsOn(t, v2)) ||
                   activeBounds(lowerBounds).anyMatch(t -> dependsOn(t, v2));
        }

        // @formatter:off
        protected FastStream<T> activeBounds(List<T> list) { return of(list); }
        private FastStream<ReferenceType> resolvedTypes(List<T> list, TypeSubstApplier subst) { return activeBounds(list).map(e -> getResolvedType(e, subst)); }
        public FastStream<ReferenceType> equalTypes(TypeSubstApplier subst) { return resolvedTypes(equalBounds, subst); }
        public FastStream<ReferenceType> upperTypes(TypeSubstApplier subst) { return resolvedTypes(upperBounds, subst); }
        public FastStream<ReferenceType> lowerTypes(TypeSubstApplier subst) { return resolvedTypes(lowerBounds, subst); }
        // @formatter:on

        public void reverseUppers() { // required for correct glbJavac behavior
            Util.reverse(upperBounds);
        }
    }

    protected static class SimpleVarBounds extends VarBounds<ReferenceType> {

        public SimpleVarBounds(BoundSet owner) {
            super(owner);
        }

        @Override
        protected ReferenceType makeBound(ReferenceType t) {
            return t;
        }

        @Override
        protected ReferenceType getType(ReferenceType bound) {
            return bound;
        }
    }

    private String varNameSuffix() {
        if (parent == null) return "";

        return parent.varNameSuffix() + nestedIndex;
    }

    protected class InferenceVar extends TypeVariable {

        public final String name;
        public final TypeParameter param;

        public InferenceVar(String name, TypeParameter param) {
            this.name = name;
            this.param = param;
        }

        // @formatter:off
        @Override public String getName() { return name + varNameSuffix(); }
        @Override public String toString() { return getName(); }
        @Override public ReferenceType getUpperBound() { throw new UnsupportedOperationException(); }
        @Override public ReferenceType getSuperType() { return TypeSystem.objectType(param); }
        // @formatter:on
    }

    protected interface InferenceVarMapper extends TypeMapper {

        @Override
        default ReferenceType mapType(ReferenceType type) {
            return type instanceof InferenceVar ? mapParam((InferenceVar) type) : type;
        }

        ReferenceType mapParam(InferenceVar param);
    }

    protected static class ResolveFailedException extends RuntimeException {

        public ResolveFailedException(String message) {
            super(message);
        }

        @Override
        public synchronized Throwable fillInStackTrace() {
            setStackTrace(new StackTraceElement[0]);
            return this;
        }
    }
}
