package net.covers1624.coffeegrinder.bytecode.transform.transformers.generics;

import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.insns.Cast;
import net.covers1624.coffeegrinder.bytecode.insns.MethodDecl;
import net.covers1624.coffeegrinder.bytecode.insns.MethodReference;
import net.covers1624.coffeegrinder.type.*;
import net.covers1624.coffeegrinder.type.TypeSubstitutions.TypeSubstApplier;
import net.covers1624.quack.collection.FastStream;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.BiConsumer;

import static java.util.Objects.requireNonNull;
import static net.covers1624.coffeegrinder.type.TypeSystem.objectType;

/**
 * Created by covers1624 on 11/10/22.
 */
public class TypeHintBoundSet extends BoundSet {

    private boolean resolved;

    public TypeHintBoundSet(Iterable<TypeParameter> typeParameters, AType retType) {
        super(typeParameters, retType);
        for (TypeParameter param : typeParameters) {
            boundsFor(param).defaultBoundsAdded();
        }
    }

    @Override
    protected InferenceVarMapper solve() {
        resolved = true;
        return super.solve();
    }

    @Override
    protected void assignable(Instruction expr, ReferenceType t) {
        // do we need to make this optional??
        if (expr instanceof Cast cast) {
            assignable(cast.getArgument(), t);
            return;
        }

        super.assignable(expr, t);
    }

    @Override
    protected void subtype(ReferenceType s, ReferenceType t) {
        if (s instanceof InferenceVar) {
            boundsFor((InferenceVar) s).incorporateDelayedAssignables(t);
        }
        else {
            // preempt some type errors. TypeHintBoundSet runs too early for all the types to actually line up sensibly all the time.
            if (t instanceof ArrayType && !(s instanceof ArrayType)) return;
            if (t instanceof WildcardType) return;
        }

        super.subtype(s, t);
    }

    @Override
    protected void retTypeAssignable(MethodDecl lambda, ReferenceType retType) {
        // It's too early here to do lambda ret type assignability.
//        super.retTypeAssignable(lambda, retType);
    }

    @Override
    protected void assignable(MethodDecl lambda, ReferenceType t) {
        if (t instanceof TypeParameter) return;

        if (t instanceof InferenceVar) {
            boundsFor((InferenceVar) t).assigned(lambda);
        }

        super.assignable(lambda, t);
    }

    @Override
    protected void assignable(MethodReference mref, ReferenceType t) {
        if (t instanceof InferenceVar) {
            boundsFor((InferenceVar) t).assigned(mref);
        }

        super.assignable(mref, t);
    }

    @Override
    protected void failProperSubtype(ReferenceType s, ReferenceType t) {
    }

    @Override
    protected void failProperTypesNotEqual(ReferenceType s, ReferenceType t) {
    }

    @Override
    protected Map<InferenceVar, ReferenceType> solveVars(List<InferenceVar> vars, InferenceVarMapper solved) {
        Map<InferenceVar, ReferenceType> solution = solvePhases(vars, solved);
        for (InferenceVar var : vars) {
            solution.putIfAbsent(var, WildcardType.createExtends(objectType(var)));
        }
        return solution;
    }

    @Override
    protected @Nullable ReferenceType solvePhase(ResolutionPhase phase, VarBounds<?> bounds, InferenceVarMapper solved) {
        TypeHintVarBounds hintBounds = (TypeHintVarBounds)bounds;
        assert hintBounds.wildcardSolution == null;

        ReferenceType t = super.solvePhase(phase, bounds, solved);
        if (t == null) return null;

        if (phase == ResolutionPhase.LOWER) {
            ReferenceType upper = solvePhase(ResolutionPhase.UPPER, bounds, solved);
            if (upper == null || TypeSystem.isObject(upper)) {
                hintBounds.wildcardSolution = WildcardType.createSuper(t);
            } else {
                // making a wildcard which is super with an upper bound is... dodgy, but works for assignability checks
                hintBounds.wildcardSolution = new WildcardType(upper, t) {
                    @Override
                    public String getFullName() {
                        return "TypeHint Wildcard[" + super.getFullName() + " extends " + getUpperBound().getFullName() + "]";
                    }
                };
            }
        } else if (phase == ResolutionPhase.UPPER) {
            hintBounds.wildcardSolution = WildcardType.createExtends(t);
        } else {
            hintBounds.wildcardSolution = t;
        }
        return t;
    }

    @Nullable
    public ReferenceType solveAndApplyTo(ReferenceType type) {
        if (failure != null)
            return null;

        InferenceVarMapper sln;
        try {
            sln = solve();
        } catch (ResolveFailedException ex) {
            return null;
        }

        TypeSubstitutions.TypeParamMapper mapper = (p) -> {
            InferenceVar var = vars.get(p);
            if (var == null) return p;

            ReferenceType r = sln.mapParam(var);
            if (r instanceof CapturedTypeVar) {
                r = WildcardType.createExtends(objectType(r));
            }
            if (!(r instanceof WildcardType)) {
                r = requireNonNull(boundsFor(p).wildcardSolution);
            }
            return r;
        };

        // Replace T with wildcards which represent the bounds which satisfy the constraints.
        // Only want to replace T with wildcards in generic types, but unfortunately this is hard to do cleanly
        // eg List<T> ->  List<? extends ...> or List<? super ...>
        type = TypeSubstitutions.subst(type, mapper);

        // A direct type usage is basically equivalent to ? extends T, due to assignability
        if (type instanceof WildcardType) {
            return type.getUpperBound();
        }

        return type;
    }

    // @formatter:off
    @Override protected TypeHintVarBounds newVarBounds() { return new TypeHintVarBounds(this); }
    @Override protected TypeHintVarBounds boundsFor(InferenceVar var) { return (TypeHintVarBounds) super.boundsFor(var); }
    @Override protected TypeHintVarBounds boundsFor(TypeParameter param) { return (TypeHintVarBounds) super.boundsFor(param); }
    @Override protected BoundSet makeNestedBoundSet(Iterable<TypeParameter> vars, AType retType) { return new TypeHintBoundSet(vars, retType); }
    // @formatter:on

    private static class TypeHintVarBounds extends SimpleVarBounds {

        private int nDefaultEq = 0;
        private int nDefaultUpper = 0;
        private int nDefaultLower = 0;

        private final List<BiConsumer<TypeHintBoundSet, ReferenceType>> delayedAssignables = new ArrayList<>(0);

        public @Nullable ReferenceType wildcardSolution;

        public TypeHintVarBounds(TypeHintBoundSet owner) {
            super(owner);
        }

        @Override
        public TypeHintBoundSet getRoot() {
            return (TypeHintBoundSet) super.getRoot();
        }

        // @formatter:off
        @Override public FastStream<ReferenceType> equalTypes(TypeSubstApplier subst) { return super.equalTypes(subst).skip(getRoot().resolved ? nDefaultEq : 0); }
        @Override public FastStream<ReferenceType> upperTypes(TypeSubstApplier subst) { return super.upperTypes(subst).skip(getRoot().resolved ? nDefaultUpper : 0); }
        @Override public FastStream<ReferenceType> lowerTypes(TypeSubstApplier subst) { return super.lowerTypes(subst).skip(getRoot().resolved ? nDefaultLower : 0); }
        // @formatter:on

        public void defaultBoundsAdded() {
            nDefaultEq = equalBounds.size();
            nDefaultUpper = upperBounds.size();
            nDefaultLower = lowerBounds.size();
        }

        public void assigned(MethodReference mref) { delayedAssignables.add((b, t) -> b.assignable(mref, t)); }

        public void assigned(MethodDecl lambda) { delayedAssignables.add((b, t) -> b.assignable(lambda, t)); }

        public void incorporateDelayedAssignables(ReferenceType t) {
            for (BiConsumer<TypeHintBoundSet, ReferenceType> f : delayedAssignables) {
                f.accept(getRoot(), t);
            }
        }
    }
}
