package net.covers1624.coffeegrinder.type.asm;

import net.covers1624.coffeegrinder.type.*;
import org.jetbrains.annotations.Nullable;
import org.objectweb.asm.Opcodes;
import org.objectweb.asm.signature.SignatureVisitor;

/**
 * Created by covers1624 on 24/12/21.
 */
public abstract class SignatureParser extends SignatureVisitor {

    protected final TypeResolver typeResolver;
    protected final ITypeParameterizedMember scope;
    private int currParamIndex;
    @Nullable
    private AsmTypeParameter currentTypeParameter;

    protected SignatureParser(TypeResolver typeResolver, ITypeParameterizedMember scope) {
        super(Opcodes.ASM9);
        this.typeResolver = typeResolver;
        this.scope = scope;
    }

    @Override
    public void visitFormalTypeParameter(String name) {
        currentTypeParameter = (AsmTypeParameter) scope.getTypeParameters().get(currParamIndex++);
        assert currentTypeParameter.getName().equals(name);
    }

    @Override
    public SignatureVisitor visitClassBound() {
        return new ReferenceTypeSignatureParser(typeResolver, scope) {
            @Override
            public void endType(AType type) {
                assert currentTypeParameter != null;
                currentTypeParameter.upperBound = (ReferenceType) type;
            }
        };
    }

    @Override
    public SignatureVisitor visitInterfaceBound() {
        assert currentTypeParameter != null;
        ReferenceType baseType = currentTypeParameter.upperBound;

        if (baseType == null) {
            // make sure that if the type parameter has no interfaces, we do actually give it the (implicit) upper bound of object
            currentTypeParameter.upperBound = typeResolver.resolveClass(TypeResolver.OBJECT_TYPE);
        }
        return new ReferenceTypeSignatureParser(typeResolver, scope) {
            @Override
            public void endType(AType type) {
                currentTypeParameter.upperBound = baseType == null
                        ? (ClassType) type
                        : TypeSystem.intersection(baseType, (ClassType) type);
            }
        };
    }

    //@formatter:off
    @Override public SignatureVisitor visitParameterType() { throw new IllegalStateException(); }
    @Override public SignatureVisitor visitReturnType() { throw new IllegalStateException(); }
    @Override public SignatureVisitor visitExceptionType() { throw new IllegalStateException(); }
    @Override public void visitBaseType(char descriptor) { throw new IllegalStateException(); }
    @Override public void visitTypeVariable(String name) { throw new IllegalStateException(); }
    @Override public SignatureVisitor visitArrayType() { throw new IllegalStateException(); }
    @Override public void visitClassType(String name) { throw new IllegalStateException(); }
    @Override public void visitInnerClassType(String name) { throw new IllegalStateException(); }
    @Override public void visitTypeArgument() { throw new IllegalStateException(); }
    @Override public SignatureVisitor visitTypeArgument(char wildcard) { throw new IllegalStateException(); }
    @Override public void visitEnd() { throw new IllegalStateException(); }
    //@formatter:on

}
