package net.covers1624.coffeegrinder.util.resolver;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;

/**
 * Capable of reading J9+ JMod files.
 * <p>
 * Created by covers1624 on 16/6/22.
 */
public class JModResolver extends JarResolver {

    // Expected JMOD Major/Minor
    private static final int JMOD_MAJOR = 0x01;
    private static final int JMOD_MINOR = 0x00;
    private static final byte[] JMOD_HEADER = {
            0x4A, 0x4D, // JM
            JMOD_MAJOR, JMOD_MINOR // 1.0
    };

    public JModResolver(Path path) {
        super(path, "/classes");
    }

    @Override
    protected void validate() {
        if (Files.notExists(path)) throw new IllegalStateException("File does not exist.");
        if (!Files.isRegularFile(path)) throw new IllegalStateException("Not a file.");
        if (!path.getFileName().toString().endsWith(".jmod")) throw new IllegalStateException("Expected jmod extension.");

        try (InputStream is = Files.newInputStream(path)) {
            byte[] magic = new byte[4];
            int len = is.read(magic);
            if (len != 4) throw new IllegalStateException("Invalid JMOD file.");
            for (int i = 0; i < 4; i++) {
                if (magic[i] != JMOD_HEADER[i]) {
                    if (i < 2) {
                        throw new IllegalStateException("Invalid JMOD file.");
                    }
                    // We only support JMOD version 1.0, if the spec changes, we may have to do something different.
                    throw new IllegalStateException("Unsupported JMOD version: " + magic[2] + "." + magic[3]);
                }
            }
        } catch (IOException ex) {
            throw new IllegalArgumentException("Failed to read JMOD file header.", ex);
        }
    }

    @Override
    public String toString() {
        return "JModResolver(" + path + ")";
    }
}
