package net.covers1624.coffeegrinder.util.resolver;

import it.unimi.dsi.fastutil.objects.Object2BooleanMap;
import it.unimi.dsi.fastutil.objects.Object2BooleanMaps;
import it.unimi.dsi.fastutil.objects.Object2BooleanOpenHashMap;
import net.covers1624.coffeegrinder.asm.ASMClassTransformer;
import org.jetbrains.annotations.Nullable;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Created by covers1624 on 8/4/21.
 */
public class ClassResolver {

    private final Map<String, CachedClassNode> classNodeCache = new ConcurrentHashMap<>();
    private final Object2BooleanMap<String> availabilityCache = Object2BooleanMaps.synchronize(new Object2BooleanOpenHashMap<>());

    private final List<ASMClassTransformer> transformers = List.of();

    @Nullable
    private Resolver targetResolver = null;
    private final List<Resolver> resolvers = new ArrayList<>();

    /**
     * Adds a resolver for a decompilation target.
     *
     * @param target The path to add.
     * @throws IllegalArgumentException Thrown when an unknown
     *                                  file type is provided.
     */
    public void setTarget(Path target) {
        //TODO support multiple targets.
        if (targetResolver != null) throw new IllegalStateException("Target already set, merge your jars.");
        setTarget(Resolver.findResolver(target));
    }

    public void setTarget(Resolver resolver) {
        targetResolver = resolver;
    }

    /**
     * Adds a Resolver to the list.
     *
     * @param path The path to add.
     * @throws IllegalArgumentException Thrown when an unknown
     *                                  file type is provided.
     */
    public void addResolver(Path path) {
        if (Files.notExists(path)) return;

        resolvers.add(Resolver.findResolver(path));
    }

    /**
     * Add all the provided paths as resolvers.
     *
     * @param paths The paths to add.
     * @throws IllegalArgumentException Thrown when an unknown
     *                                  file type is provided.
     */
    public void addResolvers(Iterable<Path> paths) {
        for (Path path : paths) {
            addResolver(path);
        }
    }

    public Resolver getTargetResolver() {
        if (targetResolver == null) throw new IllegalStateException("No target set.");
        return targetResolver;
    }

    public boolean classExists(String cName) {
        return availabilityCache.computeIfAbsent(cName, (String name) -> {
            if (targetResolver != null && targetResolver.hasClass(name)) return true;

            for (Resolver resolver : resolvers) {
                if (resolver.hasClass(name)) return true;
            }
            return false;
        });
    }

    public byte @Nullable [] getBytes(String cName) throws IOException {
        cName = cName.replace(".", "/");
        if (!availabilityCache.getOrDefault(cName, true)) return null;

        byte[] bytes = null;
        if (targetResolver != null) {
            // Target may be unset for unit tests.
            bytes = targetResolver.getClassBytes(cName);
        }

        if (bytes == null) {
            for (Resolver resolver : resolvers) {
                bytes = resolver.getClassBytes(cName);

                if (bytes != null) break;
            }
        }
        availabilityCache.put(cName, bytes != null);
        return bytes;
    }

    /**
     * Gets a {@link CachedClassNode} from this {@link ClassResolver}.
     *
     * @param cName The Class name to load.
     * @return The {@link CachedClassNode} or null if the class could not be loaded.
     * @throws RuntimeException If an error occurred whilst loading the class. TODO refine this throwable.
     */
    public CachedClassNode getClassNode(String cName) {
        if (!availabilityCache.getOrDefault(cName, true)) throw new ClassNotFoundException(cName);
        synchronized (cName.intern()) {
            CachedClassNode node = classNodeCache.get(cName);
            if (node == null) {
                try {
                    byte[] bytes = getBytes(cName);
                    if (bytes == null) throw new ClassNotFoundException(cName);

                    node = new CachedClassNode(bytes, transformers);
                    classNodeCache.put(cName, node);
                } catch (IOException e) {
                    throw new RuntimeException("Unable to construct CachedClassNode.", e);
                }
            }
            return node;
        }
    }

    public void reset() {
        availabilityCache.clear();
        classNodeCache.clear();
    }

    @Deprecated//TODO, dont throw this.
    public static class ClassNotFoundException extends RuntimeException {

        public ClassNotFoundException(String cName) {
            super(cName);
        }
    }

}
