package net.covers1624.coffeegrinder.type;

import net.covers1624.coffeegrinder.type.asm.AnnotationParser;
import net.covers1624.coffeegrinder.type.asm.TypeAnnotationParser;
import net.covers1624.quack.collection.FastStream;
import org.jetbrains.annotations.Nullable;
import org.objectweb.asm.TypeReference;
import org.objectweb.asm.tree.AnnotationNode;
import org.objectweb.asm.tree.TypeAnnotationNode;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by covers1624 on 6/3/23.
 */
public class AnnotationSupplier {

    @SuppressWarnings ("ConstantConditions")
    public static final AnnotationSupplier EMPTY = new AnnotationSupplier(null, FastStream.of(), FastStream.of());

    private final TypeResolver typeResolver;
    private final List<AnnotationNode> annotations;
    private final List<TypeAnnotationNode> typeAnnotations;

    public AnnotationSupplier(TypeResolver typeResolver, Iterable<AnnotationNode> annotations, Iterable<TypeAnnotationNode> typeAnnotations) {
        this.typeResolver = typeResolver;
        this.annotations = FastStream.of(annotations).toList();
        this.typeAnnotations = FastStream.of(typeAnnotations).toList();
    }

    public List<AnnotationData> getAnnotations() {
        return getAnnotations(TypeAnnotationData.EMPTY);
    }

    public List<AnnotationData> getAnnotations(TypeAnnotationData excludes) {
        List<AnnotationData> annotations = new ArrayList<>(this.annotations.size());
        parseAnnotations(annotations, excludes);
        return annotations;
    }

    public void parseAnnotations(List<AnnotationData> destination, TypeAnnotationData excludes) {
        AnnotationParser.parseNodes(typeResolver, excludes.getLeftMost(), annotations, ann -> {
            if (!destination.contains(ann)) {
                destination.add(ann);
            }
        });
    }

    public TypeAnnotationData getTypeAnnotations(TypeAnnotationLocation location, AType type) {
        return getTypeAnnotations(location, type, 0);
    }

    public TypeAnnotationData getTypeAnnotations(TypeAnnotationLocation location, @Nullable AType type, int index) {
        if (typeAnnotations.isEmpty()) return TypeAnnotationData.EMPTY;

        TypeAnnotationData rootNode = new TypeAnnotationData();
        parseTypeAnnotations(rootNode, location, type, index);
        return rootNode;
    }

    public void parseTypeAnnotations(TypeAnnotationData rootNode, TypeAnnotationLocation location, @Nullable AType type, int index) {
        for (TypeAnnotationNode node : typeAnnotations) {
            if (location.filter.matches(new TypeReference(node.typeRef), index)) {
                TypeAnnotationParser.parseTypeAnnotation(typeResolver, type, node, rootNode);
            }
        }
    }

    public boolean isEmpty() {
        return annotations.isEmpty() && typeAnnotations.isEmpty();
    }

    public boolean equals(AnnotationSupplier other) {
        return annotations.equals(other.annotations) && typeAnnotations.equals(other.typeAnnotations);
    }

    public enum TypeAnnotationLocation {
        // @formatter:off
        CLASS_EXTENDS       ((t, i) -> t.getSort() == TypeReference.CLASS_EXTENDS && t.getSuperTypeIndex() == -1),
        CLASS_INTERFACE     ((t, i) -> t.getSort() == TypeReference.CLASS_EXTENDS && t.getSuperTypeIndex() == i),
        METHOD_RETURN       ((t, i) -> t.getSort() == TypeReference.METHOD_RETURN),
        METHOD_RECEIVER     ((t, i) -> t.getSort() == TypeReference.METHOD_RECEIVER),
        METHOD_THROWS       ((t, i) -> t.getSort() == TypeReference.THROWS && t.getExceptionIndex() == i),
        PARAMETER           ((t, i) -> t.getSort() == TypeReference.METHOD_FORMAL_PARAMETER && t.getFormalParameterIndex() == i),
        LOCAL_VARIABLE      ((t, i) -> t.getSort() == TypeReference.LOCAL_VARIABLE),
        FIELD               ((t, i) -> t.getSort() == TypeReference.FIELD),
        TYPE_PARAMETER      ((t, i) -> (t.getSort() == TypeReference.CLASS_TYPE_PARAMETER || t.getSort() == TypeReference.METHOD_TYPE_PARAMETER) && t.getTypeParameterIndex() == i),
        TYPE_PARAMETER_BOUND((t, i) -> (t.getSort() == TypeReference.CLASS_TYPE_PARAMETER_BOUND || t.getSort() == TypeReference.METHOD_TYPE_PARAMETER_BOUND) && t.getTypeParameterIndex() == i);
        // @formatter:on

        public final LocationFilter filter;

        TypeAnnotationLocation(LocationFilter filter) {
            this.filter = filter;
        }
    }

    public interface LocationFilter {

        boolean matches(TypeReference ref, int index);
    }
}
