package net.covers1624.coffeegrinder.bytecode.transform.transformers.generics;

import net.covers1624.coffeegrinder.bytecode.IndexedInstructionCollection;
import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.SimpleInsnVisitor;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import net.covers1624.coffeegrinder.bytecode.transform.ClassTransformContext;
import net.covers1624.coffeegrinder.bytecode.transform.TopLevelClassTransformer;
import net.covers1624.coffeegrinder.debug.Step;
import net.covers1624.coffeegrinder.type.*;
import net.covers1624.coffeegrinder.util.None;
import net.covers1624.quack.collection.ColUtils;
import net.covers1624.quack.collection.FastStream;
import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.Nullable;

import java.util.LinkedList;
import java.util.List;
import java.util.function.Consumer;

import static java.util.Objects.requireNonNull;
import static net.covers1624.quack.util.SneakyUtils.notPossible;

// 4 operations to perform
// Generify - Replaces RawClass and RawMethod with generic equivalents. These may have un-inferred or un-bound type parameters after this stage
// Bind - propagate type args from the target of a field access or invoke to the result type (eg List<T>.get(int) -> T)
// Infer - Select type args for Invoke and New
// Cleanup - Remove redundant generic casts, convert raw casts to unchecked casts, insert raw casts where the generic types are now incompatible

// Bind requires the target to be fully transformed (Cleanup) in order to bind on the correct type.
// Infer requires arguments to be bound, but not inferred, (if the arguments are poly expressions) to perform nested inference
//   After infer, a smaller cleanup must be run on the nested tree
//
// This implies an order:
// Visit depth-first. Initial level: Cleanup
// Invoke/New/MethodReference
//   If !AlreadyBound
//     Visit target with level Cleanup
//     Bind
//     Visit args with level Bind
//     Break if desired level is Bind
//   If !AlreadyInferred
//     Infer poly tree
//   Visit args with level Cleanup

/**
 * Created by covers1624 on 2/4/22.
 */
public class GenericTransform extends SimpleInsnVisitor<GenericTransform.ReturnTypeInfo> implements TopLevelClassTransformer {

    private enum Pass {
        BIND(true, false),
        CLEANUP(false, true),
        FULL(true, true);

        public final boolean bind;
        public final boolean cleanup;

        Pass(boolean bind, boolean cleanup) {
            this.bind = bind;
            this.cleanup = cleanup;
        }
    }

    // TODO J17, Record.
    public record ReturnTypeInfo(
            @Nullable AType type,
            @Nullable ReferenceType cast,
            @Nullable ReferenceType explicitTypeHint,
            Pass pass
    ) {

        public static final ReturnTypeInfo NONE = new ReturnTypeInfo(null, null, null, Pass.FULL);

        public static ReturnTypeInfo assignedTo(AType type, Pass pass) {
            assert !(type instanceof ReferenceType) || TypeSystem.isFullyDefined((ReferenceType) type);
            return new ReturnTypeInfo(type, null, null, pass);
        }

        public static ReturnTypeInfo explicitHint(ReferenceType hint) {
            assert TypeSystem.isFullyDefined(hint);
            return new ReturnTypeInfo(null, null, hint, Pass.FULL);
        }

        public static ReturnTypeInfo preBind() {
            return new ReturnTypeInfo(null, null, null, Pass.BIND);
        }

        @Nullable
        @Contract (pure = true)
        public AType expectedType() {
            return explicitTypeHint != null ? explicitTypeHint : type;
        }

        public ReturnTypeInfo withCast(ReferenceType cast) {
            return new ReturnTypeInfo(type, cast, combineTypeHint(cast), pass);
        }

        private ReferenceType combineTypeHint(ReferenceType rawHintType) {
            ReferenceType t = (ReferenceType) expectedType();
            if (t == null) return rawHintType;

            if (TypeSystem.isAssignableTo(t, rawHintType)) {
                return t;
            }

            // if type is generic, attempt to recover an explicitTypeHint by combining the generics
            // type = List<? extends Number>
            // cast = ArrayList
            // explicitTypeHint = ArrayList<? extends Number>

            return BoundSet.getHierarchyCompatibleType(rawHintType, t);
        }

        public ReturnTypeInfo wrapArrayTarget(ArrayType rawArrayType) {
            AType t = expectedType();
            return new ReturnTypeInfo(null, null, t != null ? rawArrayType.withElementType(t) : null, pass);
        }
    }

    private ClassTransformContext ctx;

    @Override
    public void transform(ClassDecl cInsn, ClassTransformContext ctx) {
        this.ctx = ctx;
        cInsn.accept(new SimpleInsnVisitor<>() {
            @Override
            public None visitClassDecl(ClassDecl classDecl, None ctx) {
                visitClass(classDecl);
                return super.visitClassDecl(classDecl, ctx);
            }
        });
    }

    @Override
    public None visitClassDecl(ClassDecl classDecl, ReturnTypeInfo ret) {
        return NONE;
    }

    private void visitClass(ClassDecl classDecl) {
        boolean shouldPush = classDecl.getClazz().getDeclType() != ClassType.DeclType.TOP_LEVEL;
        if (shouldPush) {
            ctx.pushStep(classDecl.getClazz().getName(), Step.StepContextType.CLASS);
        }
        visitDefault(classDecl, ReturnTypeInfo.NONE);
        if (shouldPush) {
            ctx.popStep();
        }
    }

    @Override
    public None visitComparison(Comparison comparison, ReturnTypeInfo ret) {
        if (ret.pass.bind) {
            comparison.getLeft().accept(this, ReturnTypeInfo.NONE);
            comparison.getRight().accept(this, ReturnTypeInfo.NONE);
        }
        return NONE;
    }

    @Override
    public None visitCompoundAssignment(CompoundAssignment comp, ReturnTypeInfo ret) {
        if (ret.pass.bind) {
            comp.getReference().accept(this, ReturnTypeInfo.NONE);
            AType varType = comp.getReference().getType();
            comp.getValue().accept(this, ReturnTypeInfo.assignedTo(varType, Pass.FULL));
        }
        return NONE;
    }

    @Override
    public None visitMethodDecl(MethodDecl methodDecl, ReturnTypeInfo ret) {
        if (methodDecl.getParent() instanceof ClassDecl) {
            assert ret == ReturnTypeInfo.NONE;
            ctx.pushStep(methodDecl.getMethod().getName(), Step.StepContextType.METHOD);
            methodDecl.setReturnType(methodDecl.getMethod().getReturnType());
            super.visitMethodDecl(methodDecl, ReturnTypeInfo.NONE);
            ctx.popStep();
            return NONE;
        }

        // lambdas
        if (ret.pass.cleanup) {
            ReferenceType fInterfaceType = (ReferenceType) ret.type;
            if (fInterfaceType == null || !TypeSystem.areErasuresEqual(fInterfaceType, methodDecl.getResultType())) {
                fInterfaceType = methodDecl.getResultType();

                ClassType cType = ((ClassType) fInterfaceType).getDeclaration();
                ReturnTypeInfo hint = explicitTargetTypeHint(cType, requireNonNull(cType.getFunctionalInterfaceMethod()).getReturnType(), methodDecl.getReturnType(), null, null);
                if (hint.explicitTypeHint != null) {
                    fInterfaceType = hint.explicitTypeHint;
                }

                ctx.pushStep("Add explicit lambda type cast");
                // what fInterfaceType has return type methodDecl.getReturnType
                methodDecl.replaceWith(new Cast(methodDecl, fInterfaceType));
                ctx.popStep();
            }

            // already done
            if (!isParameterized(methodDecl.getResultType())) {
                visitLambda(methodDecl, fInterfaceType);
            }

            visitDefault(methodDecl, ReturnTypeInfo.NONE);
        }

        return NONE;

    }

    public void visitLambda(MethodDecl lambda, ReferenceType fInterfaceType) {
        ctx.pushStep("Bind lambda type " + lambda.getMethod().getName());
        lambda.setResultType(fInterfaceType);

        Method fMethod = requireNonNull(fInterfaceType.getFunctionalInterfaceMethod());
        AType retType = BoundSet.upper(fMethod.getReturnType());
        lambda.setReturnType(retType);

        List<ParameterVariable> params = lambda.parameters.filterNot(ParameterVariable::isImplicit).toList();
        for (int i = 0; i < params.size(); i++) {
            ParameterVariable param = params.get(i);
            AType paramType = param.getType();
            AType mParamType = fMethod.getParameters().get(i).getType();
            if (mParamType instanceof WildcardType w) {
                mParamType = w.isSuper() ? w.getLowerBound() : w.getUpperBound();
            }

            AType c = mParamType instanceof TypeParameter ? mParamType :
                    BoundSet.getHierarchyCompatibleType(paramType, mParamType);

            param.setType(c);
        }

        FastStream.concat(lambda.parameters, lambda.variables).forEach(
                variable -> visitLocalVariable(variable, ReturnTypeInfo.NONE)
        );

        for (Return retInsn : lambda.getReturns()) {
            retInsn.getValue().accept(this, ReturnTypeInfo.assignedTo(retType, Pass.BIND));
        }
        ctx.popStep();
    }

    @Override
    public None visitFieldDecl(FieldDecl fieldDecl, ReturnTypeInfo ret) {
        assert ret == ReturnTypeInfo.NONE;
        AType fieldType = fieldDecl.getField().getType();
        fieldDecl.getValue().accept(this, ReturnTypeInfo.assignedTo(fieldType, Pass.FULL));
        wrapWithUncheckedCast(fieldDecl.getValue(), fieldType);
        return NONE;
    }

    @Override
    public None visitCheckCast(Cast cast, ReturnTypeInfo ret) {
        if (cast.getType() instanceof ReferenceType) {
            ret = ret.withCast((ReferenceType) cast.getType());
        }

        cast.getArgument().accept(this, ret);

        if (ret.pass.cleanup) {
            cleanupCast(cast, ret);
        }

        return NONE;
    }

    private void cleanupCast(Cast cast, ReturnTypeInfo ret) {
        // The only reason for a cast to object to exist, is because the reader inserted it.
        if (TypeSystem.isObject(cast.getType())) return;

        Instruction target = cast.getArgument();
        if (target instanceof Cast || target instanceof LdcNull) return;

        AType resultType = target.getResultType();
        if (!(resultType instanceof ReferenceType)) return;

        if (TypeSystem.isAssignableTo(resultType, cast.getType())) {
            if (cast.getTag() != GenericTransformInference.REQUIRED_CAST_TAG) {
                ctx.pushStep("Remove redundant cast");
                cast.replaceWith(target);
                ctx.popStep();
            }
            return;
        }

        if (ret.explicitTypeHint == null) return;

        if (isParameterized(ret.explicitTypeHint)) {
            ReferenceType safeGenericSubtype = BoundSet.makeRepresentable(BoundSet.getHierarchyCompatibleType((ReferenceType) cast.getType(), (ReferenceType) resultType));
            if (TypeSystem.isAssignableTo(safeGenericSubtype, ret.explicitTypeHint)) {
                ctx.pushStep("Add known generics to cast");
                cast.setType(safeGenericSubtype);
                ctx.popStep();
                return;
            }
        }

        if (isRepresentable(cast, ret.explicitTypeHint) && TypeSystem.isCastableTo((ReferenceType) resultType, ret.explicitTypeHint, true)) {
            ctx.pushStep("Add unchecked generics to cast");
            cast.setType(ret.explicitTypeHint);
            ctx.popStep();
        }
    }

    private boolean isParameterized(AType type) {
        return type instanceof ParameterizedClass || type instanceof ArrayType && isParameterized(((ArrayType) type).getElementType());
    }

    @Override
    public None visitLoad(Load load, ReturnTypeInfo ret) {
        if (ret.pass.bind) {
            load.getReference().accept(this, ret);
        }
        return NONE;
    }

    @Override
    public None visitLocalVariable(LocalVariable localVariable, ReturnTypeInfo ret) {
        assert ret == ReturnTypeInfo.NONE;
        if (localVariable.getGenericSignature() != null) {
            ctx.pushStep("Replace local type " + localVariable.getName());
            localVariable.setType(getVariableGenericType(localVariable, ctx.getTypeResolver()));
            ctx.popStep();
        }
        return super.visitLocalVariable(localVariable, ReturnTypeInfo.NONE);
    }

    @Override
    public None visitPostIncrement(PostIncrement postIncrement, ReturnTypeInfo ret) {
        if (ret.pass.bind) {
            postIncrement.getReference().accept(this, ReturnTypeInfo.NONE);
        }
        return NONE;
    }

    @Override
    public None visitInstanceOf(InstanceOf instanceOf, ReturnTypeInfo ret) {
        if (ret.pass.bind) {
            instanceOf.getArgument().accept(this, ReturnTypeInfo.NONE);
        }
        return NONE;
    }

    @Override
    public None visitInvoke(Invoke invoke, ReturnTypeInfo ret) {
        if (ret.pass.bind) {
            if (!(invoke.getTarget() instanceof Nop)) {
                invoke.getTarget().accept(this, explicitTargetTypeHint(invoke.getMethod().getDeclaration(), ret, invoke.getArguments()));
            }

            // bind target based generics (including clone and getClass)
            bindMethod(invoke);

            List<Parameter> parameters = invoke.getMethod().getParameters();
            IndexedInstructionCollection<Instruction> arguments = invoke.getArguments();
            for (int i = 0; i < arguments.size(); i++) {
                AType paramType = parameters.get(i).getType();
                arguments.get(i).accept(this, ReturnTypeInfo.assignedTo(explicitArgumentTypeHint(invoke, paramType, ret), Pass.BIND));
            }
        }

        if (ret.pass.cleanup) {
            if (requiresInference(invoke.getMethod())) {
                GenericTransformInference.infer(ret, invoke, this, ctx);
            }

            List<Parameter> parameters = invoke.getMethod().getParameters();
            IndexedInstructionCollection<Instruction> arguments = invoke.getArguments();
            for (int i = 0; i < arguments.size(); i++) {
                AType paramType = parameters.get(i).getType();
                arguments.get(i).accept(this, ReturnTypeInfo.assignedTo(paramType, Pass.CLEANUP));
            }

            rawCastTargetIfNecessary(invoke);

            parameters = invoke.getMethod().getParameters();
            arguments = invoke.getArguments();
            for (int i = 0; i < arguments.size(); i++) {
                AType paramType = parameters.get(i).getType();
                wrapWithUncheckedCast(arguments.get(i), paramType);
            }
        }

        return NONE;
    }

    private static boolean requiresInference(Method method) {
        return method.isConstructor() && !TypeSystem.isFullyDefined(method.getDeclaringClass()) || !TypeSystem.isFullyDefined(method);
    }

    private void rawCastTargetIfNecessary(Invoke invoke) {
        Method method = invoke.getMethod();
        if (method.isStatic() || !anyArgsAssignedToIncompatibleTypeVariables(method.getParameters(), invoke.getArguments())) return;

        ctx.pushStep("Raw cast target of " + method.getName());
        ClassType rawTargetType = method.getDeclaringClass().asRaw();
        if (invoke.getTarget() instanceof Cast cast) {
            cast.setType(rawTargetType);
        } else if (invoke.getTarget() instanceof New newInsn) {
            // Might stuff up nested inference, but oh well, so sad
            newInsn.setMethod(newInsn.getMethod().getDeclaration().asRaw());
        } else {
            invoke.getTarget().replaceWith(new Cast(invoke.getTarget(), rawTargetType));
        }
        method = method.getDeclaration().asRaw();
        invoke.setMethod(method);

        List<Parameter> parameters = method.getParameters();
        IndexedInstructionCollection<Instruction> arguments = invoke.getArguments();
        for (int i = 0; i < parameters.size(); i++) {
            Instruction arg = arguments.get(i);
            if (!(arg instanceof Cast argCast)) continue;

            if (argCast.getTag() != GenericTransformInference.REQUIRED_CAST_TAG) continue;

            ReferenceType argType = (ReferenceType) argCast.getArgument().getResultType();
            if (TypeSystem.isAssignableTo(argType, parameters.get(i).getType())) {
                // if the argument is assignable to the parameter now, assume that the reason poly inference failed was because the required type was previously a capture.
                // Could check the previous param type to be more strict if necessary
                ctx.pushStep("Remove poly inference guard cast");
                argCast.replaceWith(argCast.getArgument());
                ctx.popStep();
            }
        }

        ctx.popStep();
    }

    private boolean anyArgsAssignedToIncompatibleTypeVariables(List<Parameter> parameters, IndexedInstructionCollection<Instruction> arguments) {
        for (int i = 0; i < parameters.size(); i++) {
            AType pType = parameters.get(i).getType();
            Instruction arg = arguments.get(i);
            if (pType instanceof TypeVariable && !TypeSystem.isAssignableTo(arg.getResultType(), pType)) {
                ReferenceType bound = ((ReferenceType) pType).getLowerBound();
                if (bound == NullConstantType.INSTANCE || bound instanceof TypeVariable && !(bound instanceof TypeParameter)) {
                    return true;
                }
            }
        }

        return false;
    }

    private static ReturnTypeInfo explicitTargetTypeHint(Field f, ReturnTypeInfo ret) {
        return explicitTargetTypeHint(f.getDeclaringClass(), f.getType(), ret, null, null);
    }

    private ReturnTypeInfo explicitTargetTypeHint(Method m, ReturnTypeInfo ret, IndexedInstructionCollection<Instruction> arguments) {
        ClassType cdecl = m.getDeclaringClass();
        if (!TypeSystem.isGeneric(cdecl)) return ReturnTypeInfo.NONE;

        // Check the method args for any relation to class type params
        LinkedList<Consumer<TypeHintBoundSet>> argBounds = new LinkedList<>();
        List<Parameter> parameters = m.getParameters();
        for (int i = 0; i < parameters.size(); i++) {
            AType paramType = parameters.get(i).getType();
            if (BoundSet.mentionsTypeParamFromClassOrOuter(paramType, cdecl)) {
                Instruction arg = arguments.get(i);

                // pre-bind the argument to get decent type info for nested inference
                arg.accept(this, ReturnTypeInfo.preBind());
                argBounds.add(b -> b.constrainAssignable(arg, paramType));
            }
        }

        return explicitTargetTypeHint(cdecl, m.getReturnType(), ret,
                argBounds.isEmpty() ? null : b -> argBounds.forEach(e -> e.accept(b)),
                argBounds.isEmpty() ? null : m.getTypeParameters()
        );
    }

    private static ReturnTypeInfo explicitOuterTargetTypeHint(ClassType type, ReturnTypeInfo ret) {
        return explicitTargetTypeHint(type.getEnclosingClass().orElseThrow(notPossible()), TypeSystem.makeThisType(type), ret, null, null);
    }

    private static ReturnTypeInfo explicitTargetTypeHint(ClassType decl, AType retType, ReturnTypeInfo ret,
            @Nullable Consumer<TypeHintBoundSet> extraConstraints, @Nullable Iterable<TypeParameter> extraTypeParams) {
        return explicitTargetTypeHint(decl, retType, ret.expectedType(), extraConstraints, extraTypeParams);
    }

    private static ReturnTypeInfo explicitTargetTypeHint(ClassType decl, AType retType, @Nullable AType expectedRetType,
            @Nullable Consumer<TypeHintBoundSet> extraConstraints, @Nullable Iterable<TypeParameter> extraTypeParams) {

        assert decl == decl.getDeclaration();
        if (!TypeSystem.isGeneric(decl)) return ReturnTypeInfo.NONE;

        boolean retTypeRelatesToClassGenerics = expectedRetType != null && (BoundSet.mentionsTypeParamFromClassOrOuter(retType, decl) || isInnerOf(retType, decl));
        if (!retTypeRelatesToClassGenerics && extraConstraints == null) { // the return type constraint may still interact with extraConstraints
            return ReturnTypeInfo.NONE;
        }

        // eg, List<T>.get(0) assigned to Number
        // get returns T, thus for T to be assignable to Number
        // T must be at least (loose) ? extends Number
        // and the target must be List<? extends Number>

        // similarly, you can imagine a generic class C<A> with a method <B extends A> List<B> get()
        // if c.get() is assigned to ArrayList<Number>
        // then B must be Number, and A must be ? super Number
        // and the target must be C<? super Number>

        ParameterizedClass p = (ParameterizedClass) TypeSystem.makeThisType(decl);
        FastStream<TypeParameter> params = collectAllTypeParams(decl);
        if (extraTypeParams != null) {
            params = params.concat(extraTypeParams);
        }
        TypeHintBoundSet b = new TypeHintBoundSet(params, retType);
        if (expectedRetType != null && retType instanceof ReferenceType) {
            b.constrainReturnAssignable((ReferenceType) expectedRetType);
        }
        if (extraConstraints != null) {
            extraConstraints.accept(b);
        }

        ReferenceType hintType = b.solveAndApplyTo(p);
        return hintType == null ? ReturnTypeInfo.NONE : ReturnTypeInfo.explicitHint(hintType);
    }

    private static FastStream<TypeParameter> collectAllTypeParams(ClassType type) {
        FastStream<TypeParameter> params = FastStream.of(type.getTypeParameters());
        if (TypeSystem.needsOuterParameterization(type)) {
            return params.concat(collectAllTypeParams(type.getEnclosingClass().orElseThrow(notPossible())));
        }
        return params;
    }

    private static AType explicitArgumentTypeHint(AbstractInvoke invoke, AType paramType, ReturnTypeInfo ret) {
        Method m = invoke.getMethod();
        if (!requiresInference(m) || !BoundSet.mentionsInferrableTypeParam(paramType, m)) return paramType;

        TypeHintBoundSet b = BoundSet.newBoundSet(invoke, TypeHintBoundSet::new);
        if (BoundSet.isPoly(m) && ret.expectedType() != null) {
            b.constrainReturnAssignable((ReferenceType) ret.expectedType());
        }

        ReferenceType hintType = b.solveAndApplyTo((ReferenceType) paramType);
        return hintType == null ? TypeSystem.erase((ReferenceType) paramType) : hintType;
    }

    private static boolean isInnerOf(AType type, ClassType outer) {
        if (!(type instanceof ParameterizedClass)) return false;

        ClassType pOuter = ((ParameterizedClass) type).getOuter();
        return pOuter != null && pOuter.getDeclaration() == outer;
    }

    @Override
    public None visitNew(New newInsn, ReturnTypeInfo ret) {
        Instruction target = newInsn.getTarget();
        if (ret.pass.bind) {
            IndexedInstructionCollection<Instruction> args = newInsn.getArguments();

            if (target != null) {
                target.accept(this, explicitOuterTargetTypeHint(newInsn.getResultType().getDeclaration(), ret));
            }

            Method method = newInsn.getMethod();
            if (method instanceof RawMethod || newInsn.getResultType() instanceof RawClass) {
                ctx.pushStep("bind ctor");
                method = method.getDeclaration();
                if (target != null) {
                    method = bindConstructor((ReferenceType) target.getResultType(), method);
                }
                newInsn.setMethod(method);
                ctx.popStep();
            }

            List<Parameter> parameters = method.getParameters();
            for (int i = 0; i < args.size(); i++) {
                Instruction arg = args.get(i);
                if (arg == target) continue;

                AType paramType = parameters.get(i).getType();
                arg.accept(this, ReturnTypeInfo.assignedTo(explicitArgumentTypeHint(newInsn, paramType, ret), Pass.BIND));
            }
        }

        if (ret.pass.cleanup) {// perform inference on the tree
            if (requiresInference(newInsn.getMethod())) {
                GenericTransformInference.infer(ret, newInsn, this, ctx);
            }

            IndexedInstructionCollection<Instruction> args = newInsn.getArguments();
            List<Parameter> parameters = newInsn.getMethod().getParameters();
            for (int i = 0; i < args.size(); i++) {
                if (args.get(i) == target) continue;

                AType paramType = parameters.get(i).getType();
                args.get(i).accept(this, ReturnTypeInfo.assignedTo(paramType, Pass.CLEANUP));
                wrapWithUncheckedCast(args.get(i), paramType);
            }

            if (newInsn.hasAnonymousClassDeclaration()) {
                newInsn.getAnonymousClassDeclaration().accept(this, ReturnTypeInfo.NONE);
            }
        }

        return NONE;
    }

    @Override
    public None visitNewArray(NewArray newArray, ReturnTypeInfo ret) {
        if (ret.pass.bind) {
            if (!newArray.isInitializer) {
                return super.visitNewArray(newArray, ReturnTypeInfo.NONE);
            }
            AType t = newArray.getType().getElementType();
            for (Instruction value : newArray.getValues()) {
                value.accept(this, ReturnTypeInfo.assignedTo(t, Pass.FULL));
            }
        }
        return NONE;
    }

    private Method bindConstructor(ReferenceType targetType, Method ctor) {
        ClassType type = ctor.getDeclaringClass();
        ClassType outerDecl = type.getEnclosingClass().orElseThrow(notPossible());
        if (!TypeSystem.isGeneric(outerDecl)) {
            return ctor; // no binding to do
        }

        ClassType outer = TypeSystem.findParameterization(outerDecl, targetType);
        if (outer instanceof RawClass) {
            return ctor; // todo, bind on raw type, raw ctor?
        }

        ParameterizedClass pOuter = (ParameterizedClass) outer;
        ParameterizedClass pTarget = new ParameterizedClass(pOuter, type, List.of());
        return new ParameterizedMethod(pTarget, ctor, pOuter, false);
    }

    @Override
    public None visitFieldReference(FieldReference fieldRef, ReturnTypeInfo ret) {
        if (!(fieldRef.getTarget() instanceof Nop)) {
            fieldRef.getTarget().accept(this, explicitTargetTypeHint(fieldRef.getField().getDeclaration(), ret));
        }
        bindField(fieldRef);
        return NONE;
    }

    @Override
    public None visitArrayElementReference(ArrayElementReference elemRef, ReturnTypeInfo ret) {
        elemRef.getArray().accept(this, ret.wrapArrayTarget((ArrayType) elemRef.getArray().getResultType()));
        if (ret.pass.bind) {
            elemRef.getIndex().accept(this, ReturnTypeInfo.NONE);
        }
        return NONE;
    }

    @Override
    public None visitArrayLen(ArrayLen arrayLen, ReturnTypeInfo ret) {
        if (ret.pass.bind) {
            arrayLen.getArray().accept(this, ReturnTypeInfo.NONE);
        }
        return NONE;
    }

    @Override
    public None visitBinary(Binary binary, ReturnTypeInfo ret) {
        if (ret.pass.bind) {
            super.visitBinary(binary, ReturnTypeInfo.NONE);
        }
        return NONE;
    }

    @Override
    public None visitForEachLoop(ForEachLoop forEachLoop, ReturnTypeInfo ret) {
        assert ret == ReturnTypeInfo.NONE;

        forEachLoop.getVariable().accept(this, ReturnTypeInfo.NONE);

        AType variableType = forEachLoop.getVariable().getType();
        ReferenceType iteratorTargetType;
        ReferenceType iteratorType = (ReferenceType) forEachLoop.getIterator().getResultType();
        if (!(iteratorType instanceof ArrayType arrayType)) {
            ClassType iterable = ctx.getTypeResolver().resolveClass(Iterable.class).getDeclaration();
            iteratorTargetType = new ParameterizedClass(null, iterable, List.of(WildcardType.createExtends((ReferenceType) variableType)));
        } else {
            iteratorTargetType = arrayType.withElementType(variableType);
        }

        forEachLoop.getIterator().accept(this, ReturnTypeInfo.explicitHint(iteratorTargetType));
        wrapWithUncheckedCast(forEachLoop.getIterator(), iteratorTargetType);

        forEachLoop.getBody().accept(this, ReturnTypeInfo.NONE);

        return NONE;
    }

    @Override
    public None visitStore(Store store, ReturnTypeInfo ret) {
        if (ret.pass.bind) {
            store.getReference().accept(this, ReturnTypeInfo.NONE);
            AType varType = store.getReference().getType();
            store.getValue().accept(this, ReturnTypeInfo.assignedTo(varType, Pass.FULL));
            wrapWithUncheckedCast(store.getValue(), varType);
        }
        return NONE;
    }

    @Override
    public None visitSwitch(Switch switchInsn, ReturnTypeInfo ret) {
        switchInsn.getValue().accept(this, ReturnTypeInfo.NONE);
        switchInsn.getBody().accept(this, ReturnTypeInfo.NONE);

        // Only push RTI out the yields of the switch.
        switchInsn.getYields().forEach(e -> e.accept(this, ret));

        return NONE;
    }

    @Override
    public None visitTernary(Ternary ternary, ReturnTypeInfo ret) {
        if (ret.pass.bind) {
            ternary.getCondition().accept(this, ReturnTypeInfo.assignedTo(PrimitiveType.BOOLEAN, Pass.FULL));
        }
        ternary.getTrueInsn().accept(this, ret);
        ternary.getFalseInsn().accept(this, ret);
        return NONE;
    }

    @Override
    public None visitReturn(Return ret, ReturnTypeInfo info) {
        assert info == ReturnTypeInfo.NONE;

        MethodDecl parent = ret.getMethod();
        boolean isLambda = !(parent.getParent() instanceof ClassDecl);

        AType retType = parent.getReturnType();
        ret.getValue().accept(this, ReturnTypeInfo.assignedTo(retType, isLambda ? Pass.CLEANUP : Pass.FULL));
        wrapWithUncheckedCast(ret.getValue(), retType);

        return NONE;
    }

    private void bindField(FieldReference fieldRef) {
        Field iField = fieldRef.getField();
        if (!(iField instanceof RawField)) return;
        Field field = iField.getDeclaration();
        if (!iField.isStatic()) {
            ReferenceType targetType = (ReferenceType) fieldRef.getTarget().getResultType();

            targetType = TypeSystem.findParameterization(field.getDeclaringClass(), targetType);
            if (targetType instanceof RawClass) return;

            if (targetType instanceof ParameterizedClass) {
                // need to re-run capture conversion on the target, because capture conversion on the original target type may not have applied to the supertype on which this method is declared
                field = TypeSubstitutions.applyOwnerParameterization(((ParameterizedClass) targetType).capture(), field);
            }
        }

        ctx.pushStep("Bind field " + iField.getName());
        fieldRef.setField(field);
        ctx.popStep();
    }

    private void bindMethod(Invoke invoke) {
        Method genericMethod = bindMethod(invoke.getTarget(), invoke.getMethod());
        if (genericMethod == null) return;

        ctx.pushStep("Bind method " + genericMethod.getName());
        invoke.setMethod(genericMethod);
        ctx.popStep();
    }

    @Nullable
    private Method bindMethod(Instruction target, Method iMethod) {
        if (target.getResultType() instanceof ArrayType && iMethod.getName().equals("clone")) {
            return new ArrayCloneMethod((ArrayType) target.getResultType());
        }

        if (!(iMethod instanceof RawMethod)) return null;

        Method method = iMethod.getDeclaration();
        if (target instanceof Nop || method.isStatic()) return method;

        ReferenceType targetType = (ReferenceType) target.getResultType();
        if (method.getName().equals("getClass") && method.getDescriptor().toString().equals("()Ljava/lang/Class;")) {
            targetType = TypeSystem.erase(targetType);
            return new GetClassMethod(targetType, ctx.getTypeResolver().resolveClassDecl(TypeResolver.CLASS_TYPE));
        }

        targetType = TypeSystem.findParameterization(method.getDeclaringClass(), targetType);
        if (targetType instanceof RawClass) return null;

        if (targetType instanceof ParameterizedClass) {
            // need to re-run capture conversion on the target, because capture conversion on the original target type may not have applied to the supertype on which this method is declared
            method = TypeSubstitutions.applyOwnerParameterization(((ParameterizedClass) targetType).capture(), method);
        }

        return method;
    }

    @Override
    public None visitMethodReference(MethodReference methodReference, ReturnTypeInfo ret) {
        if (ret.pass.bind) {
            methodReference.getTarget().accept(this, ReturnTypeInfo.NONE);
            bindMethod(methodReference);
        }
        if (ret.pass.cleanup) {
            // todo, unchecked casts?
        }

        return NONE;
    }

    private void bindMethod(MethodReference methodReference) {
        Method genericMethod = bindMethod(methodReference.getTarget(), methodReference.getMethod());
        if (genericMethod == null) return;

        ctx.pushStep("Bind method ref " + genericMethod.getName());
        methodReference.setMethod(genericMethod);
        ctx.popStep();
    }

    public void wrapWithUncheckedCast(Instruction arg, AType toType) {
        if (arg instanceof Nop) return;
        if (!(toType instanceof ReferenceType)) return;
        wrapWithUncheckedCast(arg, (ReferenceType) toType);
    }

    public void wrapWithUncheckedCast(Instruction arg, ReferenceType toType) {
        ReferenceType resultType = (ReferenceType) arg.getResultType();
        if (TypeSystem.isAssignableTo(resultType, toType)) return;

        if (toType.getLowerBound() == NullConstantType.INSTANCE) {
            throw new IllegalStateException("Cannot assign " + resultType + " to " + toType);
        }

        toType = toType.getLowerBound();
        ReferenceType uncheckedRawType = toType instanceof ParameterizedClass ? TypeSystem.erase(toType) : null;

        if (uncheckedRawType != null && !isRepresentable(arg, toType)) {
            toType = uncheckedRawType; // generate an unchecked cast instead
        }

        ctx.pushStep("Wrap with unchecked cast");
        if (!TypeSystem.isCastableTo(resultType, toType, true)) {
            // Sometimes people write code that will fail at runtime, because a type is not castable to another type
            // so they cast it to Object first to stop the compiler complaining, eg String s = (String)(Object)myInteger;
            if (uncheckedRawType != null && TypeSystem.isCastableTo(resultType, uncheckedRawType, true)) {
                toType = uncheckedRawType;
            } else {
                ctx.pushStep("Wrap with Object cast");
                arg = arg.replaceWith(new Cast(arg, ctx.getTypeResolver().resolveType(TypeResolver.OBJECT_TYPE)));
                ctx.popStep();
            }
        }

        arg.replaceWith(new Cast(arg, toType));
        ctx.popStep();
    }

    public static boolean isRepresentable(Instruction scope, AType type) {
        return switch (type) {
            case PrimitiveType primitiveType -> true;
            case ArrayType arrayType -> isRepresentable(scope, arrayType.getElementType());
            case ParameterizedClass parameterizedClass -> isRepresentable(scope, parameterizedClass);
            case ClassType classType -> true;
            case TypeParameter typeParameter -> typeParameterInScope(scope, typeParameter);
            case WildcardType wildcardType -> isRepresentable(scope, wildcardType);
            default -> false; // TODO Intersections
        };
    }

    private static boolean isRepresentable(Instruction scope, WildcardType wild) {
        return isRepresentable(scope, wild.isSuper() ? wild.getLowerBound() : wild.getUpperBound());
    }

    private static boolean isRepresentable(Instruction scope, ParameterizedClass clazz) {
        return ColUtils.allMatch(clazz.getTypeArguments(), arg -> isRepresentable(scope, arg));
    }

    public static boolean typeParameterInScope(Instruction insn, TypeParameter param) {
        ITypeParameterizedMember scope;
        if (insn instanceof MethodDecl mDecl && insn.getParent() instanceof ClassDecl) {
            scope = mDecl.getMethod();
        } else if (insn instanceof ClassDecl cDecl) {
            scope = cDecl.getClazz();
        } else {
            return typeParameterInScope(insn.getParent(), param);
        }

        TypeParameter inScope = scope.resolveTypeParameter(param.getName());
        return inScope == param;
    }

    public static AType getVariableGenericType(LocalVariable variable, TypeResolver typeResolver) {

        ITypeParameterizedMember scope;
        // We need the first class _or_ method. Current Instruction.getAncestorOfType calls don't really do what we want.
        Instruction insn = variable;
        while (true) {
            if (insn instanceof ClassDecl cDecl) {
                scope = cDecl.getClazz();
                break;
            }
            if (insn instanceof MethodDecl mDecl && insn.getParent() instanceof ClassDecl) {
                scope = mDecl.getMethod();
                break;
            }
            insn = insn.getParent();
        }
        return typeResolver.resolveGenericType(scope, requireNonNull(variable.getGenericSignature()));
    }
}
