package net.covers1624.coffeegrinder.bytecode;

import net.covers1624.quack.collection.FastStream;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Iterator;

import static net.covers1624.quack.util.SneakyUtils.unsafeCast;

/**
 * Represents an {@link InstructionSlot} capable of holding multiple instructions.
 * <p>
 * Created by covers1624 on 22/9/21.
 */
public class InstructionCollection<T extends Instruction> extends InstructionSlot<T> implements FastStream<T> {

    @Nullable
    InstructionSlot<T> head;
    @Nullable
    InstructionSlot<T> tail;

    private int size;

    /**
     * Create a new {@link InstructionCollection}.
     *
     * @param parent The parent owning instruction.
     */
    public InstructionCollection(Instruction parent) {
        super(parent);
    }

    @Override
    void onConnected() {
        InstructionSlot<?> slot = head;
        while (slot != null) {
            slot.onConnected();
            if (slot == tail) break;
            slot = slot.nextSibling;
        }
    }

    /**
     * Adds an instruction to the start of this collection.
     *
     * @param value The value.
     */
    public void addFirst(T value) {
        if (head == null) {
            add(value);
        } else {
            head.get().insertBefore(value);
        }
    }

    /**
     * Add an {@link Instruction} to the end of this collection.
     *
     * @param value The {@link Instruction}.
     */
    public void add(T value) {
        add(value, new CollectionSlot(parent));
    }

    public void addAllFirst(Iterable<? extends T> values) {
        Instruction pointer = null;
        for (T t : values) {
            if (pointer == null) {
                addFirst(t);
            } else {
                pointer.insertAfter(t);
            }

            pointer = t;
        }
    }

    /**
     * Adds multiple {@link Instruction}s to the end of this collection.
     *
     * @param values The {@link Instruction}s.
     */
    public void addAll(Iterable<? extends T> values) {
        for (T t : values) {
            add(t);
        }
    }

    /**
     * Returns the second to last {@link Instruction} in this
     * collection or <code>null</code>.
     *
     * @return The {@link Instruction}.
     */
    @Nullable
    public T secondToLastOrDefault() {
        if (tail == null) return null;
        if (tail == head) return null;
        return unsafeCast(tail.get().getPrevSiblingOrNull());
    }

    /**
     * Get the number of elements stored in this collection.
     *
     * @return The size.
     */
    public int size() {
        return size;
    }

    /**
     * Removes all elements from this collection.
     */
    public void clear() {
        // TODO this can probably be more efficient?
        for (Instruction insn : this) {
            insn.remove();
        }
    }

    //region Internal.
    protected void add(T value, CollectionSlot newSlot) {
        size++;
        if (head == null) {
            // No entries in collection.
            assert tail == null : "Collection in broken state, expected tail to be null";
            newSlot.prevSibling = this; // new entry's previous sibling is this collection.
            head = newSlot;
        } else {
            // Existing entries in collection. Append to end.
            assert tail != null : "Collection in broken state, expected tail to be non-null";
            tail.nextSibling = newSlot; // last entry's next is new
            newSlot.prevSibling = tail; // new insn's prev is the old tail
        }
        newSlot.nextSibling = nextSibling; // the next of the new insn is this collections next sibling
        tail = newSlot; // the tail of the collection is the new insn.

        newSlot.set(value);
    }

    @Override
    public T getValueOrNull() {
        throw new UnsupportedOperationException();
    }

    @Nullable
    @Override
    InstructionSlot<?> onNext() {
        // When someone requests to iterate into us, we either:
        // - Give them the head of our list if present.
        // - Give them our next sibling.
        if (head != null) return head.onNext();
        if (nextSibling != null) return nextSibling.onNext();
        return null;
    }

    @Nullable
    @Override
    InstructionSlot<?> onPrevious(@Nullable InstructionSlot<?> inSlot) {
        // When someone requests to iterate backwards into us, we either:
        // - Give them the tail of our list if present.
        // - Ask our previous sibling, if present.
        if (head != inSlot && tail != null) return tail.onPrevious(inSlot);
        if (prevSibling != null) return prevSibling.onPrevious(inSlot);
        return null;
    }

    @Override
    protected void checkInvariant() {
        super.checkInvariant();
        assert head == null || head.prevSibling == this;
        assert tail == null || tail.nextSibling == nextSibling;

        for (InstructionSlot<?> slot = head; slot != null; slot = slot.nextSibling) {
            slot.checkInvariant();
        }
    }

    @Override
    public void set(Instruction value) {
        throw new UnsupportedOperationException();
    }

    @Override
    public int count() {
        return size;
    }

    @Override
    public boolean isEmpty() {
        return size == 0;
    }

    @NotNull
    @Override
    public Iterator<T> iterator() {
        return new Iterator<T>() {

            @Nullable
            InstructionSlot<?> next = head;

            @Override
            public boolean hasNext() {
                return next != null;
            }

            @Override
            public T next() {
                assert next != null;
                InstructionSlot<?> curr = next;
                if (next != tail) {
                    next = next.nextSibling;
                } else {
                    next = null;
                }
                return unsafeCast(curr.get());
            }
        };
    }

    protected class CollectionSlot extends InstructionSlot<T> {

        public CollectionSlot(Instruction parent) {
            super(parent, false);
        }

        @Override
        void remove() {
            size--;
            assert value != null;
            if (head == this) {
                if (head == tail) {
                    head = null;
                    tail = null;
                } else {
                    assert nextSibling != null; // next sibling must be set, as head and tail don't match.
                    assert prevSibling == InstructionCollection.this; // We are the head, our previous should be the collection.
                    head = unsafeCast(nextSibling);
                    head.prevSibling = InstructionCollection.this;
                }
            } else if (tail == this) {
                assert prevSibling != null;
                tail = unsafeCast(prevSibling);
                prevSibling.nextSibling = nextSibling;
            } else {
                assert nextSibling != null;
                assert prevSibling != null;
                nextSibling.prevSibling = prevSibling;
                prevSibling.nextSibling = nextSibling;
            }

            parent.onChildModified();
            parent.invalidateFlags();

            // Yeet
            if (value.isConnected()) {
                value.releaseRef();
            }

            nextSibling = null;
            prevSibling = null;
        }

        @Override
        void insertBefore(Instruction value) {
            size++;
            CollectionSlot newInsn = new CollectionSlot(parent);

            if (head == this) {
                head = newInsn;
                newInsn.prevSibling = InstructionCollection.this;
            } else {
                assert prevSibling != null;
                newInsn.prevSibling = prevSibling;
                prevSibling.nextSibling = newInsn;
            }
            newInsn.nextSibling = this;
            prevSibling = newInsn;

            // Set after linking.
            newInsn.set(value);
        }

        @Override
        void insertAfter(Instruction value) {
            size++;
            CollectionSlot newInsn = new CollectionSlot(parent);

            if (tail == this) {
                tail = newInsn;
                newInsn.nextSibling = InstructionCollection.this.nextSibling;
            } else {
                assert nextSibling != null;
                newInsn.nextSibling = nextSibling;
                nextSibling.prevSibling = newInsn;
            }
            newInsn.prevSibling = this;
            nextSibling = newInsn;

            newInsn.set(value);
        }

        @Override
        protected void checkInvariant() {

            if (this == head) {
                // If we are the head, the previous must be our owning collection.
                assert prevSibling == InstructionCollection.this;
            } else {
                // Otherwise, our previous.next, should be us
                assert prevSibling != null;
                assert prevSibling.nextSibling == this;
            }

            if (this == tail) {
                // If we are the tail, our next should be our parent collections next.
                assert nextSibling == InstructionCollection.this.nextSibling;
            } else {
                // Otherwise, our next.previous, should be us.
                assert nextSibling != null;
                assert nextSibling.prevSibling == this;
            }
        }

        @Override
        boolean isInCollection() {
            return true;
        }
    }
    //endregion
}
