package net.covers1624.coffeegrinder.util.jvm;

import net.covers1624.jdkutils.JavaInstall;
import net.covers1624.jdkutils.utils.JavaPropExtractor;
import net.covers1624.quack.collection.FastStream;
import org.jetbrains.annotations.Nullable;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Created by covers1624 on 12/4/21.
 */
public class JVMUtils {

    private static final String JAVA_CLASS_PATH = "java.class.path";
    private static final String JAVA_BOOT_CLASS_PATH = "sun.boot.class.path";

    public static List<Path> getRuntimeJREClasspath() {
        // J8
        List<Path> paths = listSysProp(JAVA_BOOT_CLASS_PATH);
        if (paths == null) {
            // J9+
            String javaHomeProp = System.getProperty("java.home");
            if (javaHomeProp == null) throw new IllegalStateException("Incompatible runtime: 'java.home' system property is missing.");

            Path javaHome = Paths.get(javaHomeProp);
            paths = listJMods(javaHome);
        }
        return paths;
    }

    public static JavaInstall getJavaInstall(Path executable) {
        JavaInstall install = JavaInstall.parse(executable);
        if (install == null) throw new RuntimeException("Unable to find valid Java installation at the provided Java executable.");
        return install;
    }

    public static List<Path> getTargetJREClasspath(JavaInstall javaInstall) {
        Map<String, String> props = JavaPropExtractor.extractProperties(
                JavaInstall.getJavaExecutable(javaInstall.javaHome, true),
                Collections.singletonList(JAVA_BOOT_CLASS_PATH)
        );
        if (props == null) throw new RuntimeException("Failed to extract java classpath information.");

        List<Path> paths = listProp(props.get(JAVA_BOOT_CLASS_PATH));
        if (paths != null) {
            return paths;
        }

        return listJMods(javaInstall.javaHome);
    }

    public static List<Path> getRuntimeClasspath() {
        List<Path> paths = listSysProp(JAVA_CLASS_PATH);
        if (paths != null) {
            return paths;
        }
        return Collections.emptyList();
    }

    private static @Nullable List<Path> listSysProp(String sysProp) {
        return listProp(System.getProperty(sysProp));
    }

    private static @Nullable List<Path> listProp(@Nullable String value) {
        if (value == null || value.isEmpty()) return null;

        return FastStream.of(value.split(File.pathSeparator))
                .distinct()
                .map(Paths::get)
                .filter(Files::exists)
                .toList();
    }

    private static List<Path> listJMods(Path javaHome) {
        if (Files.notExists(javaHome)) throw new IllegalStateException("Incompatible runtime: 'java.home' system property resolves to non-existent path.");

        Path jModFolder = javaHome.resolve("jmods");
        if (Files.notExists(jModFolder)) throw new IllegalStateException("Incompatible runtime: '${java.home}/jmods' does not exist.");

        try (Stream<Path> files = Files.list(jModFolder)) {
            return files.collect(Collectors.toList());
        } catch (IOException e) {
            throw new IllegalStateException("Failed to list jmods directory.");
        }
    }
}
