package net.covers1624.coffeegrinder.util.asm;

import org.objectweb.asm.*;
import org.objectweb.asm.tree.ClassNode;
import org.objectweb.asm.tree.MethodNode;
import org.objectweb.asm.util.Textifier;
import org.objectweb.asm.util.TraceClassVisitor;
import org.objectweb.asm.util.TraceMethodVisitor;

import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;

/**
 * Created by covers1624 on 19/9/21.
 */
public class OrderedTextifier extends Textifier {

    public static void main(String[] args) throws IOException {
        Path path = Paths.get(String.join(" ", args));
        try (InputStream is = Files.newInputStream(path)) {
            ClassReader reader = new ClassReader(is);
            System.out.println(textifyClass(v -> reader.accept(v, 0)));
        }
    }

    public static String textifyClass(Consumer<ClassVisitor> func) {
        var textifier = new OrderedTextifier();
        func.accept(new TraceClassVisitor(null, textifier, null));
        return toString(textifier);
    }

    public static String textifyMethod(Consumer<MethodVisitor> func) {
        var textifier = new OrderedTextifier();
        func.accept(new TraceMethodVisitor(null, textifier));
        return toString(textifier);
    }

    public static String textify(ClassNode cNode) {
        return textifyClass(cNode::accept);
    }

    public static String textify(MethodNode mNode) {
        return textifyMethod(mNode::accept);
    }

    private static String toString(Textifier textifier) {
        var sw = new StringWriter();
        var pw = new PrintWriter(sw, true);
        textifier.print(pw);
        return sw.toString();
    }

    private final Map<Label, String> orderedLabels = new HashMap<>();

    private final List<PendingLabel> pendingLabels = new ArrayList<>();

    public OrderedTextifier() {
        super(Opcodes.ASM9);
    }

    @Override
    public void visitLabel(Label label) {
        var labelName = "L" + orderedLabels.size();
        orderedLabels.put(label, labelName);
        super.visitLabel(label);
    }

    @Override
    protected void appendLabel(Label label) {
        var name = orderedLabels.get(label);
        if (name != null) {
            stringBuilder.append(name);
            return;
        }

        // Insert a temporary label for now. Will be cleaned up at the end.
        var pending = new PendingLabel(text.size(), stringBuilder.length(), label, "PENDING_" + label);
        pendingLabels.add(pending);
        stringBuilder.append(pending.tempName);
    }

    @Override
    public void visitMethodEnd() {
        super.visitMethodEnd();
        if (pendingLabels.isEmpty()) return;

        // Do in reverse so we don't shift any indexes.
        for (var pending : pendingLabels.reversed()) {
            String line = (String) text.get(pending.line);
            if (!(line.startsWith(pending.tempName, pending.idx))) throw new IllegalStateException("Char offset invalid.");

            text.set(pending.line,
                    line.substring(0, pending.idx)
                    + orderedLabels.get(pending.label)
                    + line.substring(pending.idx + pending.tempName.length())
            );
        }
    }

    @Override
    protected Textifier createTextifier() {
        return new OrderedTextifier();
    }

    private record PendingLabel(int line, int idx, Label label, String tempName) { }
}
