package net.covers1624.coffeegrinder.bytecode.insns;

import net.covers1624.coffeegrinder.bytecode.*;
import net.covers1624.coffeegrinder.type.AType;
import net.covers1624.coffeegrinder.type.PrimitiveType;
import net.covers1624.coffeegrinder.util.EnumBitSet;
import net.covers1624.quack.collection.FastStream;
import org.jetbrains.annotations.Nullable;

import java.util.LinkedList;
import java.util.List;
import java.util.Objects;

/**
 * Created by covers1624 on 24/2/21.
 */
public final class Block extends Instruction {

    public final InstructionCollection<Instruction> instructions = new InstructionCollection<>(this);

    private boolean isEntryPoint;
    private final List<Branch> branches = new LinkedList<>();

    @Nullable
    private String name;

    public Block() {
        this(null, false);
    }

    public Block(String name) {
        this(name, false);
    }

    private Block(@Nullable String name, boolean unused) {
        this.name = name;
    }

    @Override
    public AType getResultType() {
        return PrimitiveType.VOID;
    }

    @Override
    public EnumBitSet<InstructionFlag> getDirectFlags() {
        return InstructionFlag.NONE;
    }

    @Override
    public Block copy() {
        Block block = new Block();
        block.instructions.addAll(instructions.map(Instruction::copy));
        return block;
    }

    @Override
    protected void onConnected() {
        super.onConnected();
        if (name == null) {
            name = firstAncestorOfType(MethodDecl.class).getSynBlockName(this);
        }
    }

    @Override
    public <R, C> R accept(InsnVisitor<R, C> visitor, C ctx) {
        return visitor.visitBlock(this, ctx);
    }

    public Block extractRange(Instruction first, Instruction last) {
        return extractRange(null, first, last);
    }

    /**
     * Extracts a range of {@link Instruction}s from this block into a new block.
     *
     * @param first The first instruction to move. (Inclusive)
     * @param last  The last instruction to move. (Inclusive)
     * @return THe new block.
     */
    public Block extractRange(@Nullable String subName, Instruction first, Instruction last) {
        assert first.getParent() == this;
        assert last.getParent() == this;

        Block newBlock = new Block();
        if (subName != null) {
            newBlock.setName(getSubName(subName));
        }
        newBlock.setOffsets(first);
        Instruction pointer = first;
        while (true) {
            Instruction next = pointer.getNextSiblingOrNull();
            newBlock.instructions.add(pointer);
            if (pointer == last) break;
            assert next != null; // Next may be null if you select the last sibling as the end.
            pointer = next;
        }
        return newBlock;
    }

    //@formatter:off
    void addBranch(Branch branch) { branches.add(branch); }
    void remBranch(Branch branch) { branches.remove(branch); }
    public FastStream<Branch> getBranches() { return FastStream.of(branches); }
    void setEntryPoint(boolean isEntryPoint) { this.isEntryPoint = isEntryPoint; }
    public int getIncomingEdgeCount() { return branches.size() + (isEntryPoint ? 1 : 0); }
    public String getName() { return Objects.requireNonNull(name); }
    public void setName(String name) { this.name = name; }
    //@formatter:on

    public String getSubName(String nameSuffix) {
        return getName() + "_" + nameSuffix;
    }
}
