package net.covers1624.coffeegrinder.debug;

import net.covers1624.quack.util.SneakyUtils;
import org.jetbrains.annotations.Nullable;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.function.Supplier;

/**
 * Created by covers1624 on 13/5/21.
 */
public class Step implements AutoCloseable {

    private final DebugStepper stepper;
    private final int id;
    private final StepType type;
    private final StepContextType contextType;
    private final String name;

    private Status status = Status.SUCCESS;

    @Nullable
    Throwable exception;

    @Nullable
    private final Step parent;
    final LinkedList<Step> children = new LinkedList<>();

    @Nullable
    final Supplier<String> supplier;

    private final String preStepContent;
    private final long startTime;

    @Nullable
    private String postStepContent = "NO CONTENT";
    private long endTime;
    boolean isFinished;

    // Constructs a synthetic finished micro step.
    Step(DebugStepper stepper, int id, String name, Step parent, String preContent, String postContent, long startTime, long endTime) {
        this.stepper = stepper;
        this.id = id;
        this.type = StepType.CONTENT;
        this.contextType = StepContextType.NONE;
        this.name = name;
        this.parent = parent;
        this.supplier = null;
        this.preStepContent = preContent;
        this.postStepContent = postContent;
        this.startTime = startTime;
        this.endTime = endTime;
        this.isFinished = true;
    }

    Step(DebugStepper stepper, int id, StepType type, StepContextType contextType, String name, @Nullable Step parent, @Nullable String preContent, @Nullable Supplier<String> supplier) {
        this.stepper = stepper;
        this.id = id;
        this.type = type;
        this.contextType = contextType;
        this.name = name;
        this.parent = parent;
        this.supplier = supplier;
        this.preStepContent = preContent != null ? preContent : supplier != null ? supplier.get() : "NO CONTENT";
        this.startTime = System.nanoTime();
    }

    boolean finish() {
        // Try and grab a marked exception from the Stepper.
        Throwable ex = stepper.clearExcept();
        if (ex != null) {
            // Propagate the exception to children.
            propagateThrowable(ex);
            return false;
        }

        if (supplier != null) {
            // Try and grab the content.
            try {
                postStepContent = supplier.get();
            } catch (Throwable e) {
                // If we have errored. Dump the stack trace as the content.
                if (status == Status.ERROR) {
                    StringWriter writer = new StringWriter();
                    e.printStackTrace(new PrintWriter(writer));
                    writer.flush();
                    postStepContent = "Exception was thrown whilst printing in error state.\n" + writer;
                } else {
                    // Re-throw otherwise, no non-error step should fail like this.
                    SneakyUtils.throwUnchecked(e);
                }
            }
        }

        // Mark time and finish.
        endTime = System.nanoTime();
        isFinished = true;
        return true;
    }

    // Propagate the exception deepest non-finished child.
    private void propagateThrowable(Throwable ex) {
        boolean handled = false;
        for (Step child : children) {
            if (child.isFinished) continue;
            handled = true;
            child.propagateThrowable(ex);
            break; // There should only ever be one active child.
        }
        // If a child did not 'handle' the exception, we did.
        if (!handled) {
            exception = ex;
        }
        // Mark as errored and continue with finalization.
        status = Status.ERROR;
        stepper.popStep();
    }

    /**
     * Gets a short descriptive name for this step.
     * Usually the Transformer/method name/class name
     *
     * @return The name.
     */
    public String getName() {
        return name;
    }

    public int getId() {
        return id;
    }

    /**
     * Gets this Steps type.
     *
     * @return The type of Step this is.
     */
    public StepType getType() {
        return type;
    }

    public StepContextType getContextType() {
        return contextType;
    }

    /**
     * Gets the status of this Step.
     *
     * @return The status.
     */
    public Status getStatus() {
        return status;
    }

    /**
     * Gets the exception that was thrown during the execution of this step.
     *
     * @return The exception or <code>null</code> if the step executed successfully.
     */
    @Nullable
    public Throwable getCaughtException() {
        return exception;
    }

    /**
     * Gets the parent for this step.
     * This will be null for the root node.
     *
     * @return The parent step, or null for the root node.
     */
    @Nullable
    public Step getParent() {
        return parent;
    }

    /**
     * Gets the children for this step.
     *
     * @return An Immutable view of children.
     */
    public List<Step> getChildren() {
        return Collections.unmodifiableList(children);
    }

    /**
     * The content provided for this step before any action was taken by the step.
     *
     * @return The content. May return null if the Stepper responsible does not support content.
     */
    @Nullable
    public String getPreStepContent() {
        return preStepContent;
    }

    /**
     * The content provided for this step after action was taken by the step.
     *
     * @return The content. May return null if the Stepper responsible does not support content.
     */
    @Nullable
    public String getPostStepContent() {
        return postStepContent;
    }

    /**
     * @return The time in nanoseconds this step started.
     */
    public long getStartTime() {
        return startTime;
    }

    /**
     * @return The time in nanoseconds this step ended.
     */
    public long getEndTime() {
        return endTime;
    }

    /**
     * @return The duration in nanoseconds this step took.
     */
    public long getDuration() {
        return getEndTime() - getStartTime();
    }

    @Override
    public void close() {
        stepper.popStep();
    }

    public enum StepContextType {
        NONE,
        CLASS,
        METHOD,
        TRANSFORMER,
    }

    public enum StepType {
        CONTENT,
        TIMING,
    }

    public enum Status {
        SUCCESS,
        ERROR,
    }
}
