package net.covers1624.coffeegrinder.type.asm;

import net.covers1624.coffeegrinder.type.*;
import org.jetbrains.annotations.Nullable;
import org.objectweb.asm.Opcodes;
import org.objectweb.asm.Type;
import org.objectweb.asm.signature.SignatureReader;
import org.objectweb.asm.signature.SignatureVisitor;

import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicReference;

/**
 * Created by covers1624 on 29/12/21.
 */
public abstract class ReferenceTypeSignatureParser extends SignatureVisitor {

    private final TypeResolver typeResolver;
    private final ITypeParameterizedMember scope;

    @Nullable
    private ParameterizedClass paramaterizedOuter;

    @Nullable
    private ClassType currentType;

    private final List<ReferenceType> currentArguments = new LinkedList<>();

    protected ReferenceTypeSignatureParser(TypeResolver typeResolver, ITypeParameterizedMember scope) {
        super(Opcodes.ASM9);
        this.typeResolver = typeResolver;
        this.scope = scope;
    }

    public static ReferenceType parse(TypeResolver typeResolver, ITypeParameterizedMember scope, String signature) {
        AtomicReference<ReferenceType> referenceType = new AtomicReference<>();
        new SignatureReader(signature).accept(new ReferenceTypeSignatureParser(typeResolver, scope) {
            @Override
            public void endType(AType type) {
                referenceType.set((ReferenceType) type);
            }
        });
        return Objects.requireNonNull(referenceType.get(), "Didn't parse anything??");
    }

    @Override
    public void visitBaseType(char descriptor) {
        endType(typeResolver.resolveType(Type.getType(String.valueOf(descriptor))));
    }

    @Override
    public void visitTypeVariable(String name) {
        endType(Objects.requireNonNull(scope.resolveTypeParameter(name)));
    }

    @Override
    public SignatureVisitor visitArrayType() {
        return new ReferenceTypeSignatureParser(typeResolver, scope) {
            @Override
            public void endType(AType type) {
                ReferenceTypeSignatureParser.this.endType(typeResolver.makeArray(type));
            }
        };
    }

    @Override
    public void visitClassType(String name) {
        currentType = typeResolver.resolveClassDecl(name);
    }

    @Override
    public void visitTypeArgument() {
        currentArguments.add(WildcardType.createExtends(typeResolver.resolveClass(TypeResolver.OBJECT_TYPE)));
    }

    @Override
    public void visitInnerClassType(String name) {
        assert currentType != null;
        paramaterizedOuter = new ParameterizedClass(paramaterizedOuter, currentType, List.copyOf(currentArguments));
        currentType = typeResolver.resolveClassDecl(currentType.getFullName() + "$" + name);
        currentArguments.clear();
    }

    @Override
    public SignatureVisitor visitTypeArgument(char wildcard) {
        return new ReferenceTypeSignatureParser(typeResolver, scope) {
            @Override
            public void endType(AType type) {
                switch (wildcard) {
                    case EXTENDS:
                        currentArguments.add(WildcardType.createExtends((ReferenceType) type));
                        break;
                    case SUPER:
                        currentArguments.add(WildcardType.createSuper((ReferenceType) type));
                        break;
                    case INSTANCEOF:
                        currentArguments.add((ReferenceType) type);
                        break;
                    default:
                        throw new IllegalArgumentException(wildcard + "");
                }
            }
        };
    }

    @Override
    public void visitEnd() {
        assert currentType != null;
        if (!currentArguments.isEmpty() || paramaterizedOuter != null) {
            endType(new ParameterizedClass(paramaterizedOuter, currentType, List.copyOf(currentArguments)));
            return;
        }
        if (TypeSystem.isGeneric(currentType)) {
            endType(currentType.asRaw());
            return;
        }
        endType(currentType);
    }

    public abstract void endType(AType type);
}
