package net.covers1624.coffeegrinder.debug;

import net.covers1624.coffeegrinder.bytecode.DebugPrintOptions;
import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.debug.Step.StepContextType;
import net.covers1624.coffeegrinder.source.JavaSourceVisitor;
import net.covers1624.coffeegrinder.type.TypeResolver;
import org.jetbrains.annotations.Nullable;

import java.util.function.Supplier;

/**
 * A Stepper serves a few tasks.
 * 1, It profiles the Decompiler as it processes through transformation steps.
 * 2, It stores the intermediate result of these transformation steps for use in the debugger.
 * 3, Profiles various steps the decompiler takes.
 * <p>
 * Created by covers1624 on 12/5/21.
 */
public interface Stepper {

    /**
     * Get the {@link DebugPrintOptions} for this stepper.
     *
     * @return The {@link DebugPrintOptions}.
     */
    DebugPrintOptions getOpts();

    /**
     * Pushes a new context to the Stepper.
     *
     * @param supplier The supplier to evaluate the data on each step.
     */
    void pushContext(ContextSupplier supplier);

    /**
     * Pops the current context off of the stepper.
     */
    void popContext();

    /**
     * Starts a step in the current context.
     *
     * @param name A descriptive name for the step.
     */
    @Nullable
    Step pushStep(String name);

    /**
     * Starts a step in the current context.
     *
     * @param name A descriptive name for the step.
     */
    @Nullable
    Step pushStep(String name, StepContextType contextType);

    /**
     * Start a step in the current context, overriding the current context's
     * content supplier.
     *
     * @param name       A descriptive name for the step.
     * @param preContent The content supplier for the step.
     */
    @Nullable
    Step pushStepWithContent(String name, StepContextType contextType, Supplier<String> preContent);

    /**
     * Stops a transformation step.
     */
    void popStep();

    /**
     * Starts a non-content step.
     *
     * @param name The name.
     */
    @Nullable
    Step pushTiming(String name);

    /**
     * Stops a non-content step.
     */
    void popTiming();

    /**
     * Marks the Stepper as having caught an exception.
     *
     * @param e The throwable. Will be re-throw silently.
     */
    void except(Throwable e);

    /**
     * Gets the root node for the step tree.
     *
     * @return The root node, may return <code>null</code> if no steps were taken.
     */
    @Nullable
    Step getRoot();

    interface ContextSupplier {

        String print(DebugPrintOptions options);

        static ContextSupplier ofInsn(TypeResolver resolver, Instruction insn) {
            return options -> {
                if (!options.javaSource()) {
                    return insn.toString(options);
                }

                var visitor = new JavaSourceVisitor(resolver, options.showImplicits());
                var text = insn.accept(visitor);
                return String.join("\n", text.lines);
            };
        }
    }
}
