package net.covers1624.coffeegrinder.bytecode;

import net.covers1624.coffeegrinder.util.EnumBitSet;
import net.covers1624.quack.collection.FastStream;
import org.jetbrains.annotations.Nullable;

import java.util.List;

import static org.objectweb.asm.Opcodes.*;

/**
 * Created by covers1624 on 31/8/21.
 */
public enum AccessFlag {
    PUBLIC("public"),
    PRIVATE("private"),
    PROTECTED("protected"),
    STATIC("static"),
    FINAL("final"),
    //    SUPER("super"),
    SYNCHRONIZED("synchronized"),
    OPEN("open"),
    TRANSITIVE("transitive"),
    VOLATILE("volatile"),
    BRIDGE("bridge"),
    STATIC_PHASE("static_phase"),
    VARARGS("varargs"),
    TRANSIENT("transient"),
    NATIVE("native"),
    INTERFACE("interface"),
    ABSTRACT("abstract"),
    STRICT("strict-fp"),
    ANNOTATION("annotation"),
    SYNTHETIC("synthetic"),
    ENUM("enum"),
    MANDATED("mandated"),
    MODULE("module"),
    RECORD("record"),
    ;

    public static final List<AccessFlag> ACCESS_MODIFIERS = List.of(
            PUBLIC,
            PRIVATE,
            PROTECTED
    );

    public final String name;

    AccessFlag(String name) {
        this.name = name;
    }

    public static EnumBitSet<AccessFlag> unpackClass(int packed) {
        EnumBitSet<AccessFlag> flags = EnumBitSet.noneOf(AccessFlag.class);
        if ((packed & ACC_PUBLIC) > 0) flags.set(PUBLIC);
        if ((packed & ACC_PROTECTED) > 0) flags.set(PROTECTED);
        if ((packed & ACC_FINAL) > 0) flags.set(FINAL);
//        if ((packed & ACC_SUPER) > 0) flags.set(SUPER);
        if ((packed & ACC_INTERFACE) > 0) flags.set(INTERFACE);
        if ((packed & ACC_ABSTRACT) > 0) flags.set(ABSTRACT);
        if ((packed & ACC_SYNTHETIC) > 0) flags.set(SYNTHETIC);
        if ((packed & ACC_ANNOTATION) > 0) flags.set(ANNOTATION);
        if ((packed & ACC_ENUM) > 0) flags.set(ENUM);
        if ((packed & ACC_MANDATED) > 0) flags.set(MANDATED);
        if ((packed & ACC_MODULE) > 0) flags.set(MODULE);
        if ((packed & ACC_RECORD) > 0) flags.set(RECORD);
        return flags;
    }

    public static EnumBitSet<AccessFlag> unpackInnerClass(int packed) {
        EnumBitSet<AccessFlag> flags = EnumBitSet.noneOf(AccessFlag.class);
        if ((packed & ACC_PRIVATE) > 0) flags.set(PRIVATE);
        if ((packed & ACC_STATIC) > 0) flags.set(STATIC);
        return flags;
    }

    public static EnumBitSet<AccessFlag> unpackMethod(int packed) {
        EnumBitSet<AccessFlag> flags = EnumBitSet.noneOf(AccessFlag.class);
        if ((packed & ACC_PUBLIC) > 0) flags.set(PUBLIC);
        if ((packed & ACC_PRIVATE) > 0) flags.set(PRIVATE);
        if ((packed & ACC_PROTECTED) > 0) flags.set(PROTECTED);
        if ((packed & ACC_STATIC) > 0) flags.set(STATIC);
        if ((packed & ACC_FINAL) > 0) flags.set(FINAL);
        if ((packed & ACC_SYNCHRONIZED) > 0) flags.set(SYNCHRONIZED);
        if ((packed & ACC_BRIDGE) > 0) flags.set(BRIDGE);
        if ((packed & ACC_VARARGS) > 0) flags.set(VARARGS);
        if ((packed & ACC_NATIVE) > 0) flags.set(NATIVE);
        if ((packed & ACC_ABSTRACT) > 0) flags.set(ABSTRACT);
        if ((packed & ACC_STRICT) > 0) flags.set(STRICT);
        if ((packed & ACC_SYNTHETIC) > 0) flags.set(SYNTHETIC);
        if ((packed & ACC_MANDATED) > 0) flags.set(MANDATED);
        return flags;
    }

    public static EnumBitSet<AccessFlag> unpackParameter(int packed) {
        EnumBitSet<AccessFlag> flags = EnumBitSet.noneOf(AccessFlag.class);
        if ((packed & ACC_FINAL) > 0) flags.set(FINAL);
        if ((packed & ACC_SYNTHETIC) > 0) flags.set(SYNTHETIC);
        if ((packed & ACC_MANDATED) > 0) flags.set(MANDATED);
        return flags;
    }

    public static EnumBitSet<AccessFlag> unpackField(int packed) {
        EnumBitSet<AccessFlag> flags = EnumBitSet.noneOf(AccessFlag.class);
        if ((packed & ACC_PUBLIC) > 0) flags.set(PUBLIC);
        if ((packed & ACC_PRIVATE) > 0) flags.set(PRIVATE);
        if ((packed & ACC_PROTECTED) > 0) flags.set(PROTECTED);
        if ((packed & ACC_STATIC) > 0) flags.set(STATIC);
        if ((packed & ACC_FINAL) > 0) flags.set(FINAL);
        if ((packed & ACC_VOLATILE) > 0) flags.set(VOLATILE);
        if ((packed & ACC_TRANSIENT) > 0) flags.set(TRANSIENT);
        if ((packed & ACC_SYNTHETIC) > 0) flags.set(SYNTHETIC);
        if ((packed & ACC_ENUM) > 0) flags.set(ENUM);
        if ((packed & ACC_MANDATED) > 0) flags.set(MANDATED);
        return flags;
    }

    public static String toString(EnumBitSet<AccessFlag> flags) {
        return FastStream.of(flags.toSet()).map(e -> e.name + ' ').join("");
    }

    public static String stringRep(EnumBitSet<AccessFlag> flags) {
        String str = toString(flags);
        if (flags.get(INTERFACE)) {
            str = str.replace(" abstract", "");
            str = str.replace("interface annotation", "@interface");
        }
        if (flags.get(ENUM)) {
            str = str.replace("final ", "");
            str = str.replace("abstract ", "");
        }
        str = str.replace("varargs ", "");
        return str;
    }

    @Nullable
    public static AccessFlag getAccess(EnumBitSet<AccessFlag> accessFlags) {
        for (AccessFlag flag : ACCESS_MODIFIERS) {
            if (accessFlags.get(flag)) {
                return flag;
            }
        }
        return null;
    }
}
