package net.covers1624.coffeegrinder.bytecode;

import org.jetbrains.annotations.Nullable;

import static java.util.Objects.requireNonNull;
import static net.covers1624.quack.util.SneakyUtils.unsafeCast;

/**
 * Represents a Slot to hold an {@link Instruction}.
 * <p>
 * Created by covers1624 on 21/9/21.
 */
public class InstructionSlot<T extends Instruction> {

    final Instruction parent;

    /**
     * The value stored in this slot.
     */
    @Nullable
    Instruction value;

    /**
     * This slots previous sibling.
     */
    @Nullable
    InstructionSlot<?> prevSibling;

    /**
     * This slots next sibling.
     */
    @Nullable
    InstructionSlot<?> nextSibling;

    /**
     * Create a new {@link InstructionSlot}.
     *
     * @param parent The parent owning instruction.
     */
    public InstructionSlot(Instruction parent) {
        this(parent, true);
    }

    protected InstructionSlot(Instruction parent, boolean link) {
        this.parent = parent;

        if (link) {
            // Link.
            if (this.parent.firstChild == null) {
                assert this.parent.lastChild == null : "Parent in broken state, expected lastChild to be null";
                this.parent.firstChild = this;
            } else {
                assert this.parent.lastChild != null : "Parent in broken state, expected last child to be non-null";
                this.parent.lastChild.nextSibling = this;
                prevSibling = this.parent.lastChild;
            }
            this.parent.lastChild = this;
        }
    }

    /**
     * Set the value in this slot.
     *
     * @param value The value.
     */
    public void set(Instruction value) {
        assert !parent.isDescendantOf(value);
        Instruction oldValue = this.value;
        this.value = value;

        parent.invalidateFlags();
        parent.onChildModified();

        if (parent.isConnected()) {
            linkValue();
            if (oldValue != null) {
                oldValue.releaseRef();
            }
        }
    }

    private void linkValue() {
        assert value != null;
        InstructionSlot<?> prevSlot = value.inSlot;
        value.inSlot = this;
        value.addRef();

        if (prevSlot != null && prevSlot.value == value) {
            prevSlot.remove();
        }
    }

    void onConnected() {
        linkValue();
    }

    /**
     * Return the value stored inside the slot.
     *
     * @return The value, otherwise <code>null</code>
     */
    @Nullable
    T getValueOrNull() {
        if (value == null) throw new InvalidTreeStateException("Slot is required to have a value. Was it not set or moved?");

        return unsafeCast(value);
    }

    /**
     * Return the value stored inside the slot.
     *
     * @return The value.
     */
    public T get() {
        return requireNonNull(getValueOrNull());
    }

    protected void checkInvariant() {
        // Assert linkage
        assert prevSibling == null || prevSibling.nextSibling == this;
        assert nextSibling == null || nextSibling.prevSibling == this;
        assert value == null || value.inSlot == this;
    }

    /**
     * Allows the slot to process what another slot should
     * see when using this slot as the next.
     * <p>
     * This is used by collections to descend into the nested tree.
     *
     * @return The slot, or null.
     */
    @Nullable
    InstructionSlot<?> onNext() {
        return this;
    }

    /**
     * Allows the slot to process what another slot should
     * see when using this slot as the previous.
     * <p>
     * This is used by collections to descend into the nested tree.
     *
     * @param inSlot Which slot is asking.
     * @return The slot, or null.
     */
    @Nullable
    InstructionSlot<?> onPrevious(@Nullable InstructionSlot<?> inSlot) {
        return this;
    }

    void removeValue() {
        assert value != null;
        if (value.isConnected()) {
            value.releaseRef();
        }
        value = null;
        parent.invalidateFlags();
    }

    void remove() {
        removeValue();
    }

    void insertBefore(Instruction value) {
        throw new UnsupportedOperationException("Instruction is not inside a collection.");
    }

    void insertAfter(Instruction value) {
        throw new UnsupportedOperationException("Instruction is not inside a collection.");
    }

    boolean isInCollection() {
        return false;
    }
}
