package net.covers1624.coffeegrinder.bytecode.transform.transformers;

import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.InstructionFlag;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import net.covers1624.coffeegrinder.bytecode.matching.ComparisonMatching;
import net.covers1624.coffeegrinder.bytecode.transform.MethodTransformContext;
import net.covers1624.coffeegrinder.bytecode.transform.MethodTransformer;
import net.covers1624.coffeegrinder.type.ClassType;
import net.covers1624.coffeegrinder.type.TypeResolver;
import net.covers1624.coffeegrinder.type.TypeSystem;
import org.jetbrains.annotations.Nullable;
import org.objectweb.asm.Type;

import java.util.LinkedList;
import java.util.List;

import static net.covers1624.coffeegrinder.bytecode.matching.BranchLeaveMatching.matchBranch;
import static net.covers1624.coffeegrinder.bytecode.matching.BranchLeaveMatching.matchThrow;
import static net.covers1624.coffeegrinder.bytecode.matching.IfMatching.matchNopFalseIf;
import static net.covers1624.coffeegrinder.bytecode.matching.InvokeMatching.matchInvoke;
import static net.covers1624.coffeegrinder.bytecode.matching.LoadStoreMatching.*;

/**
 * The bytecode structure of TryWithResources was changed in Java 11.
 * <p>
 * See the <a href="https://github.com/openjdk/jdk/commit/01509e5b5e7dad9ac3efa96a3b90cd0b249e6c39">commit</a>.
 * <p>
 * Created by covers1624 on 13/3/23.
 */
public class J11TryWithResourcesTransform implements MethodTransformer {

    private static final Type AUTO_CLOSEABLE = Type.getType(AutoCloseable.class);
    private static final Type THROWABLE = Type.getType(Throwable.class);

    private final ClassType autoCloseableClass;
    private final ClassType throwableClass;

    public J11TryWithResourcesTransform(TypeResolver typeResolver) {
        autoCloseableClass = typeResolver.resolveClass(AUTO_CLOSEABLE);
        throwableClass = typeResolver.resolveClass(THROWABLE);
    }

    @Override
    public void transform(MethodDecl function, MethodTransformContext ctx) {
        // 'pre-order' so we handle outer tries before inners, otherwise we could inline stuff required for the outer try.
        function.descendantsOfType(TryCatch.class).reversed().forEach(tryCatch -> {
            if (tryCatch.isConnected()) {
                transformTryWithResources(tryCatch, ctx);
            }
        });

    }

    private void transformTryWithResources(TryCatch tryCatch, MethodTransformContext ctx) {
        // Match block prior to try, contains the resource store.
        // BLOCK L0
        //     ...
        //     STORE resourceVariable ...
        //     BRANCH L1
        // BLOCK L1
        //     TRY_CATCH
        //         ...
        //     CATCH (Throwable primaryException)
        //         [nullcheck BR rethrow]
        //         TRY
        //             resourceVariable.close()
        //         CATCH (Throwable ex)
        //             primaryException.addSuppressed(ex)
        //
        //         BLOCK rethrow
        //             THROW primaryException
        //
        // Each exit of the body:
        //     [nullcheck] BR next
        //     resourceVariable.close()

        Block tryParent = (Block) tryCatch.getParent();
        Branch preTryBranch = tryParent.getBranches().onlyOrDefault();
        if (preTryBranch == null) return;
        if (!(preTryBranch.getParent() instanceof Block preTryBlock)) return;
        if (preTryBlock.getNextSiblingOrNull() != tryParent) return;

        // Match the resource, ensure its AutoClosable.
        Store resourceStore = matchStoreLocal(preTryBranch.getPrevSiblingOrNull());
        if (resourceStore == null) return;
        if (!TypeSystem.isAssignableTo(resourceStore.getResultType(), autoCloseableClass)) return;
        LocalVariable resourceVariable = resourceStore.getVariable();

        // Match the primary try-handler.
        // catch (Throwable primaryCatchEx)
        TryCatch.TryCatchHandler primaryCatch = tryCatch.handlers.onlyOrDefault();
        if (primaryCatch == null) return;
        LocalVariable primaryCatchEx = primaryCatch.getVariable().variable;
        if (!primaryCatchEx.isSynthetic() || primaryCatchEx.getType() != throwableClass) return;

        // Last block in the primary catch should re-throw the primary catch variable.
        // BLOCK primaryCatchExit
        //     STORE s_1 = LOAD primaryCatchEx
        //     THROW LOAD s_1
        Block primaryCatchExit = primaryCatch.getBody().blocks.last();
        {
            Store stackStore = matchStoreLocalLoadLocal(primaryCatchExit.getFirstChildOrNull(), primaryCatchEx);
            if (stackStore == null) return;
            if (matchThrow(stackStore.getNextSiblingOrNull(), stackStore.getVariable()) == null) return;
        }

        // First block in primary catch should branch to the immediate next block.
        // Block SYN_L0
        //     BRANCH maybeNullCheckBlock
        Block maybeNullcheckBlock = getNextBlockFromBranchBlock(primaryCatch.getBody().getEntryPoint());
        if (maybeNullcheckBlock == null || maybeNullcheckBlock == primaryCatchExit) return; // Quick bailout for catch(Throwable ex) { throw ex; }

        Block closeTryCatchBlock = maybeNullcheckBlock;

        // If the immediate next block has this store, we have null-checks in our TWR.
        Store nullCheckStoreLoad = matchStoreLocalLoadLocal(maybeNullcheckBlock.getFirstChildOrNull(), resourceVariable);
        boolean hasNullChecks = nullCheckStoreLoad != null;
        if (hasNullChecks) {
            // BLOCK maybeNullCheckBlock
            //     STORE s_1 = LOAD resourceVariable
            //     if (LOAD s_1 == null) BRANCH primaryCatchBlock
            //     BRANCH SYN_BLOCK
            IfInstruction nullCheckIf = matchNopFalseIf(nullCheckStoreLoad.getNextSiblingOrNull());
            if (nullCheckIf == null) return;
            if (ComparisonMatching.matchEqualNull(nullCheckIf.getCondition(), nullCheckStoreLoad.getVariable()) == null) return;
            if (matchBranch(nullCheckIf.getTrueInsn(), primaryCatchExit) == null) return; // True exits to the re-throw.

            // Then branches to bouncer block, then to the closeTryCatchBlock
            // BLOCK SYN_BLOCK
            //     BRANCH closeTryCatchBlock
            Block nextBlock = matchBranchToNextSiblingBlock(nullCheckIf.getNextSiblingOrNull());
            if (nextBlock == null) return;
            closeTryCatchBlock = getNextBlockFromBranchBlock(nextBlock);
            if (closeTryCatchBlock == null) return;
        }

        // Match the try-catch inside the primary catch handler.
        // BLOCK closeTryCatchBlock
        //     TRY_CATCH
        //         BLOCK first
        //             STORE s_1 = LOAD resourceVariable
        //             INVOKE.VIRTUAL s_1.close()
        //             BRANCH next
        //         BLOCK next
        //             BRANCH closeTryCatchBlock
        //     CATCH (Throwable var_1)
        //         BLOCK first
        //             BRANCH suppressedBlock
        //         BLOCK suppressedBlock
        //             STORE s_1 = LOAD primaryCatchEx
        //             STORE s_2 = LOAD var1
        //             INVOKE.VIRTUAL s_1.addSuppressed(s_2)
        //             BRANCH closeTryCatchBlock
        if (!(closeTryCatchBlock.instructions.onlyOrDefault() instanceof TryCatch closeTryCatch)) return;
        {
            Block first = closeTryCatch.getTryBody().blocks.firstOrDefault();
            if (first == null) return;
            Store store = matchStoreLocalLoadLocal(first.getFirstChildOrNull(), resourceVariable);
            if (store == null) return;
            Instruction invoke = store.getNextSiblingOrNull();
            if (invoke == null) return;
            if (!matchAutoCloseableCloseInvoke(invoke, store.getVariable())) return;

            Block next = matchBranchToNextSiblingBlock(invoke.getNextSiblingOrNull());
            if (next == null) return;
            if (matchBranch(next.instructions.onlyOrDefault(), primaryCatchExit) == null) return;
        }
        TryCatch.TryCatchHandler closeCatchHandler = closeTryCatch.handlers.onlyOrDefault();
        if (closeCatchHandler == null) return;
        {
            Block suppressedBlock = getNextBlockFromBranchBlock(closeCatchHandler.getBody().getEntryPointOrNull());
            if (suppressedBlock == null) return;
            Store s1 = matchStoreLocalLoadLocal(suppressedBlock.getFirstChildOrNull(), primaryCatchEx);
            if (s1 == null) return;
            Store s2 = matchStoreLocalLoadLocal(s1.getNextSiblingOrNull(), closeCatchHandler.getVariable().variable);
            if (s2 == null) return;
            Invoke invoke = matchInvoke(s2.getNextSiblingOrNull(), Invoke.InvokeKind.VIRTUAL, "addSuppressed");
            if (invoke == null) return;
            if (invoke.getArguments().size() != 1) return;
            if (matchLoadLocal(invoke.getTarget(), s1.getVariable()) == null) return;
            if (matchLoadLocal(invoke.getArguments().firstOrDefault(), s2.getVariable()) == null) return;
            if (matchBranch(invoke.getNextSiblingOrNull(), primaryCatchExit) == null) return;
        }

        // We successfully matched the Try With Resources pattern.
        ctx.pushStep("produce try-with-resources");
        TryWithResources twr = tryCatch.replaceWith(new TryWithResources(resourceStore, tryCatch.getTryBody()).withOffsets(tryCatch));
        ctx.popStep();

        // Now that we have matched the pattern. We need to clean up its exits.
        twr.getTryBody()
                .descendantsOfType(Branch.class)
                .filter(e -> !e.getTargetBlock().isDescendantOf(twr))
                .forEach(exit -> {
                    ctx.pushStep("process exit " + exit.getTargetBlock().getName());
                    processExit(exit, hasNullChecks, resourceVariable, ctx);
                    ctx.popStep();
                });

        if (resourceVariable.isSynthetic() && resourceVariable.getLoadCount() == 0) {
            ctx.pushStep("simplify to resource reference");
            resourceStore.replaceWith(resourceStore.getValue());
            ctx.popStep();
        }
    }

    private static void processExit(Branch exitBranch, boolean hasNullChecks, LocalVariable variable, MethodTransformContext ctx) {
        Block exit = exitBranch.getTargetBlock();
        // Exits may have a null check around the resource close. We do the following:
        // - Remove the null check if it exists, and inline the next stage.
        // - Remove the close invoke.
        // - If the only thing left is a branch, inline it.
        // - If the only thing left is a load of a synthetic local + return, inline it.

        // Starts the same regardless of null check or not.
        // STORE stackStore = LOAD resourceVariable
        Store stackStore = matchStoreLocalLoadLocal(exit.getFirstChildOrNull(), variable);
        if (stackStore == null) return;
        if (hasNullChecks) {

            // If we have null checks match it.
            // if (LOAD stackStore == null) BRANCH realExitBranch
            // BRANCH next
            IfInstruction nullCheckIf = matchNopFalseIf(stackStore.getNextSiblingOrNull());
            if (nullCheckIf == null) return;
            if (ComparisonMatching.matchEqualNull(nullCheckIf.getCondition(), stackStore.getVariable()) == null) return;
            Block next = matchBranchToNextSiblingBlock(nullCheckIf.getNextSiblingOrNull());
            if (next == null) return;
            if (next.getIncomingEdgeCount() != 1) return;

            // Ensure our true exit branches over the block we suspect the resource close is inside.
            // BLOCK next
            //     ...
            //     BRANCH realExitBranch
            if (!(nullCheckIf.getTrueInsn() instanceof Branch realExitBranch)) return;
            if (!(next.getLastChild() instanceof Branch nextBranchToRealExit)) return;
            if (nextBranchToRealExit.getTargetBlock() != realExitBranch.getTargetBlock()) return;

            // Javac puts the convergence of the null check branches (one around the resource close) outside the try.
            // Pull it in if it's only a branch otherwise we would produce a goto and potentially mess up fallthrough detection.
            var convergedExit = realExitBranch.getTargetBlock();
            if ((convergedExit.getFirstChild() instanceof Branch)) {
                ctx.pushStep("Inline null-check exit");
                next.insertAfter(convergedExit);
                ctx.popStep();
            }

            // Match the first instruction inside the suspected close block, it should be identical to what we had for the null-check block.
            // STORE stackStore = LOAD resourceVariable
            Store realExitStore = matchStoreLocalLoadLocal(next.getFirstChild(), variable);
            if (realExitStore == null) return;
            stackStore = realExitStore;

            // yeet, Inline the next block into us.
            ctx.pushStep("strip null check from exit");
            exit.instructions.clear();
            exit.instructions.addAll(next.instructions);
            next.remove();
            ctx.popStep();
        }
        // Match the close invoke
        // INVOKE.VIRTUAL stackStore.close()
        Instruction invoke = stackStore.getNextSiblingOrNull();
        if (invoke == null) return;
        if (!matchAutoCloseableCloseInvoke(invoke, stackStore.getVariable())) return;

        // Yeet just the store and invoke.
        ctx.pushStep("strip resource close from exit");
        stackStore.remove();
        invoke.remove();
        ctx.popStep();

        // Now we have removed the null-check, resource close
        // we can check to see if the remaining contents is a load from a synthetic local, to a return.

        // If the exit is owned only by the twr, do we try and inline things.
        if (exit.getIncomingEdgeCount() != 1) return;

        // We may have one or more intermediate blocks we need to match through for our return pattern.
        // There will be an intermediate block for each TWR we are inside, as each will have had their own
        // resource closure on the exit.
        Instruction effectiveAfterExit = exit.getFirstChildOrNull();
        List<Block> branchBlocks = new LinkedList<>();
        while (effectiveAfterExit instanceof Branch realExitBranch) {
            // Can't do anything here.
            if (realExitBranch.getTargetBlock().getIncomingEdgeCount() != 1) return;
            var realExitBlock = realExitBranch.getTargetBlock();
            effectiveAfterExit = realExitBlock.getFirstChildOrNull();
            branchBlocks.add(realExitBlock);
        }

        // Try inline synthetic local return.
        // BLOCK exit
        //     STORE s_1 = LOAD synLocal
        //     RETURN s_1
        Store synStore = matchStoreLocal(effectiveAfterExit);
        if (synStore == null) return;
        Load loadLocal = matchLoadLocal(synStore.getValue());
        if (loadLocal == null) return;
        LocalVariable synLocal = loadLocal.getVariable();
        if (!synLocal.isSynthetic()) return;
        if (synLocal.getReferenceCount() != 2) return;
        if (!(synStore.getNextSiblingOrNull() instanceof Return ret)) return;
        if (matchLoadLocal(ret.getValue(), synStore.getVariable()) == null) return;

        // Where our exit branch comes from should have a store to the above synthetic local.
        // STORE synLocal = ...
        Store storeSynLocal = matchStoreLocal(exitBranch.getPrevSiblingOrNull(), synLocal);
        if (storeSynLocal == null) return;

        // Replace the store to the synthetic local with a return.
        // Remove the exit branch, and the exit block.
        ctx.pushStep("inline synthetic local return");
        storeSynLocal.replaceWith(new Return(ret.getMethod(), storeSynLocal.getValue()).withOffsets(ret));
        exitBranch.remove();
        branchBlocks.forEach(Instruction::remove);
        exit.remove();
        ctx.popStep();
    }

    // Matches a block which only branches to the immediate next sibling.
    // Returns the sibling block.
    @Nullable
    private static Block getNextBlockFromBranchBlock(@Nullable Block block) {
        if (block == null) return null;
        return matchBranchToNextSiblingBlock(block.instructions.onlyOrDefault());
    }

    // Matches a branch which points to the immediate next sibling block.
    // Returns the sibling block.
    @Nullable
    private static Block matchBranchToNextSiblingBlock(@Nullable Instruction insn) {
        if (insn == null) return null;
        if (!(insn.getParent() instanceof Block parent)) return null;

        if (!(parent.getNextSiblingOrNull() instanceof Block next)) return null;
        if (matchBranch(insn, next) == null) return null;

        return next;
    }

    // TODO copied from old TryWithResourcesTransform, perhaps we should extend and make things protected.
    private static boolean matchAutoCloseableCloseInvoke(@Nullable Instruction insn, LocalVariable autoCloseable) {
        if (!(insn instanceof Invoke invoke)) return false;
        if (matchLoadLocal(invoke.getTarget(), autoCloseable) == null) return false;
        if (invoke.getKind() != Invoke.InvokeKind.VIRTUAL && invoke.getKind() != Invoke.InvokeKind.INTERFACE) return false;
        if (!invoke.getMethod().getName().equals("close")) return false;
        return invoke.getMethod().getParameters().isEmpty();
    }
}
