package net.covers1624.coffeegrinder.bytecode;

import net.covers1624.coffeegrinder.bytecode.insns.*;
import org.jetbrains.annotations.Nullable;

import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;

/**
 * Created by covers1624 on 11/24/25.
 */
public class ScopeVisitor<R, C> extends InsnVisitor<R, C> {

    private record Scope(@Nullable Scope prev, @Nullable LocalVariable var) { }

    private Scope scope = new Scope(null, null);

    private final InsnVisitor<R, C> inner;

    public ScopeVisitor(InsnVisitor<R, C> inner) {
        this.inner = inner;
    }

    public boolean currentScopeHasDeclarations() {
        return scope.var != null;
    }

    public R visit(Instruction insn, C ctx) {
        if (insn.getParentOrNull() instanceof IfInstruction) {
            return withScope(() -> insn.accept(this, ctx));
        }

        return insn.accept(this, ctx);
    }

    @Override
    public R visitDefault(Instruction insn, C ctx) {
        return insn.accept(inner, ctx);
    }

    @Override
    public R visitMethodDecl(MethodDecl methodDecl, C ctx) {
        return withScope(() -> {
            methodDecl.parameters.forEach(this::declare);
            // declare all the params
            return super.visitMethodDecl(methodDecl, ctx);
        });
    }

    @Override
    public R visitBlockContainer(BlockContainer container, C ctx) {
        return visitWithScope(container, ctx);
    }

    @Override
    public R visitSwitchSection(SwitchTable.SwitchSection switchSection, C ctx) {
        return visitWithScope(switchSection, ctx);
    }

    @Override
    public R visitForLoop(ForLoop loop, C ctx) {
        return visitWithScope(loop, ctx);
    }

    @Override
    public R visitForEachLoop(ForEachLoop forEachLoop, C ctx) {
        return visitWithScope(forEachLoop, ctx);
    }

    @Override
    public R visitTryCatchHandler(TryCatch.TryCatchHandler catchHandler, C ctx) {
        return visitWithScope(catchHandler, ctx);
    }

    @Override
    public R visitTryWithResources(TryWithResources tryWithResources, C ctx) {
        return visitWithScope(tryWithResources, ctx);
    }

    private R visitWithScope(Instruction insn, C ctx) {
        return withScope(() -> visitDefault(insn, ctx));
    }

    private R withScope(Supplier<R> func) {
        scope = new Scope(scope, null);
        try {
            return func.get();
        } finally {
            while (scope.var != null) { scope = requireNonNull(scope.prev); }
            scope = requireNonNull(scope.prev);
        }
    }

    @Override
    public R visitLocalReference(LocalReference local, C ctx) {
        var r = super.visitLocalReference(local, ctx);

        if (local.isConnected() && !local.isReadFrom() && !isDeclared(local.variable)) {
            declare(local.variable);
        }

        return r;
    }

    private void declare(LocalVariable var) {
        scope = new Scope(scope, var);
    }

    public boolean isDeclared(LocalVariable var) {
        var s = scope;
        while (s != null) {
            if (s.var == var) {
                return true;
            }

            s = s.prev;
        }

        return false;
    }

    public boolean isDeclared(String name) {
        return getVariableInScope(name) != null;
    }

    public @Nullable LocalVariable getVariableInScope(String name) {
        var s = scope;
        while (s != null) {
            if (s.var != null && s.var.getName().equals(name)) {
                return s.var;
            }

            s = s.prev;
        }

        return null;
    }
}
