package net.covers1624.coffeegrinder.bytecode.matching;

import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import org.jetbrains.annotations.Nullable;

import static net.covers1624.coffeegrinder.bytecode.matching.LdcMatching.matchLdcInt;

/**
 * Created by covers1624 on 22/11/21.
 */
public class AssignmentMatching {

    /**
     * Match the given Instruction to a {@link CompoundAssignment} of the given operation.
     *
     * @param insn The Instruction to match.
     * @param op   The Kind of operation.
     * @return The {@link CompoundAssignment} or <code>null</code>.
     */
    @Nullable
    public static CompoundAssignment matchCompoundAssignment(@Nullable Instruction insn, BinaryOp op) {
        if (!(insn instanceof CompoundAssignment assignment) || assignment.getOp() != op) return null;

        return assignment;
    }

    /**
     * Match the given Instruction to a {@link CompoundAssignment} of the given operation
     * with the given {@link LdcNumber} value.
     *
     * @param insn The Instruction to match.
     * @param op   The Kind of operation.
     * @param ldc  The {@link LdcNumber} value.
     * @return The {@link CompoundAssignment} or <code>null</code>.
     */
    @Nullable
    public static CompoundAssignment matchCompoundAssignment(@Nullable Instruction insn, BinaryOp op, int ldc) {
        CompoundAssignment assignment = matchCompoundAssignment(insn, op);
        if (assignment == null) return null;

        if (matchLdcInt(assignment.getValue(), ldc) == null) return null;

        return assignment;
    }

    /**
     * Match the given Instruction to a {@link CompoundAssignment} type with:
     * <pre>
     *  - Either ADD or SUB operation
     *  - Int value of 1
     * </pre>
     *
     * @param insn The Instruction to match.
     * @return The {@link CompoundAssignment} or <code>null</code>.
     */
    @Nullable
    public static CompoundAssignment matchPreInc(@Nullable Instruction insn) {
        if (!(insn instanceof CompoundAssignment assignment)) return null;
        if (assignment.getOp() != BinaryOp.ADD && assignment.getOp() != BinaryOp.SUB) return null;

        if (!(assignment.getValue() instanceof LdcNumber ldc) || ldc.getValue().doubleValue() != 1) return null;

        return assignment;
    }

    @Nullable
    public static Binary matchStoreArgBinaryWithPossibleCast(Store store) {
        Instruction value = store.getValue();

        if (value instanceof Cast c) {
            if (c.getType() == store.getReference().getType()) {
                value = c.getArgument();
            }
        }

        if (!(value instanceof Binary binary)) return null;
        return binary;
    }
}
