package net.covers1624.coffeegrinder.bytecode;

import net.covers1624.coffeegrinder.bytecode.insns.ClassDecl;
import net.covers1624.coffeegrinder.bytecode.insns.tags.InsnTag;
import net.covers1624.coffeegrinder.source.AstSourceVisitor;
import net.covers1624.coffeegrinder.source.LineBuffer;
import net.covers1624.coffeegrinder.type.AType;
import net.covers1624.coffeegrinder.util.EnumBitSet;
import net.covers1624.coffeegrinder.util.None;
import net.covers1624.quack.collection.FastStream;
import net.covers1624.quack.util.Copyable;
import org.jetbrains.annotations.MustBeInvokedByOverriders;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.lang.reflect.AccessFlag;
import java.util.*;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;

import static java.util.Objects.requireNonNull;
import static net.covers1624.quack.collection.FastStream.of;
import static net.covers1624.quack.util.SneakyUtils.unsafeCast;

/**
 * Represents an Instruction within an AST tree.
 * <p>
 * Created by covers1624 on 21/9/21.
 */
public abstract class Instruction implements Copyable<Instruction> {

    private int refCount;

    @Nullable
    private EnumBitSet<InstructionFlag> flags = InstructionFlag.INVALID_FLAGS;

    @Nullable
    InstructionSlot<?> inSlot;

    @Nullable
    InstructionSlot<?> firstChild;
    @Nullable
    InstructionSlot<?> lastChild;

    @Nullable
    private InsnTag tag;

    private int bytecodeOffset = -1;
    private int sourceLine = -1;

    protected Instruction() {
    }

    /**
     * Gets the stack type of the value produced by this {@link Instruction}.
     *
     * @return The {@link AType}.
     */
    public abstract AType getResultType();

    /**
     * Deep copy this instruction.
     * <p>
     * This many not be supported by all instructions.
     * <p>
     * Implementors should use copy-constructor syntax.
     *
     * @return The copied instruction.
     * @throws UnsupportedOperationException If this instruction does not support being copied.
     */
    public Instruction copy() {
        throw new UnsupportedOperationException("Instruction '" + getClass().getName() + "' does not support being copied.");
    }

    /**
     * Any direct flags for this {@link Instruction}.
     * <p>
     * The return value should be cached statically inside the {@link Instruction} impl.
     *
     * @return An {@link EnumBitSet} representing the flags.
     * @see #getFlags()
     */
    public abstract EnumBitSet<InstructionFlag> getDirectFlags();

    /**
     * Gets the flags describing the behavior of this {@link Instruction}.
     * This method wil compute the flags on-demand and caches the
     * result, until some changes the AST invalidate the cache.
     *
     * @return An {@link EnumBitSet} representing the flags.
     * @see #getDirectFlags()
     */
    public final EnumBitSet<InstructionFlag> getFlags() {
        if (flags == InstructionFlag.INVALID_FLAGS) {
            flags = computeFlags();
        }
        return flags;
    }

    /**
     * Returns weather this instruction exposes this specific flag either
     * directly, or from one of its children.
     *
     * @param flag The flag to check.
     * @return If the flag is present.
     */
    public final boolean hasFlag(InstructionFlag flag) {
        return getFlags().get(flag);
    }

    /**
     * Returns weather this instruction exposes this specific flag directly.
     *
     * @param flag The flag to check.
     * @return If the flag is present.
     */
    public final boolean hasDirectFlag(InstructionFlag flag) {
        return getDirectFlags().get(flag);
    }

    /**
     * Pass this {@link Instruction} through the supplied {@link InsnVisitor}.
     *
     * @param visitor The {@link InsnVisitor}.
     * @param ctx     The context.
     * @return The return result from the {@link InsnVisitor}.
     */
    public abstract <R, C> R accept(InsnVisitor<R, C> visitor, C ctx);

    /**
     * Passes this {@link Instruction} through the supplied {@link InsnVisitor}.
     * This method will assume the generic of {@link None} for the Context
     * generic provided by {@link InsnVisitor}
     *
     * @param visitor The {@link InsnVisitor}.
     * @return The return result from the {@link InsnVisitor}.
     */
    public final <R> R accept(InsnVisitor<R, None> visitor) {
        return accept(visitor, None.INSTANCE);
    }

    /**
     * The {@link Instruction} which this instruction is a child of.
     *
     * @return The parent {@link Instruction} or <code>null</code>
     * @see #getParent()
     */
    @Nullable
    public final Instruction getParentOrNull() {
        return inSlot != null ? inSlot.parent : null;
    }

    /**
     * The {@link Instruction} which this instruction is a child of.
     *
     * @return The parent {@link Instruction}.
     * @see #getParentOrNull()
     */
    public final Instruction getParent() {
        return requireNonNull(getParentOrNull());
    }

    /**
     * This {@link Instruction}s next sibling instruction.
     *
     * @return The next sibling {@link Instruction} or <code>null</code>.
     * @see #getNextSibling()
     */
    @Nullable
    public final Instruction getNextSiblingOrNull() {
        if (inSlot == null || inSlot.nextSibling == null) return null;
        InstructionSlot<?> slot = inSlot.nextSibling.onNext();
        return slot == null ? null : slot.getValueOrNull();
    }

    /**
     * This {@link Instruction}s next sibling instruction.
     *
     * @return The next sibling {@link Instruction}.
     * @see #getNextSiblingOrNull()
     */
    public final Instruction getNextSibling() {
        return requireNonNull(getNextSiblingOrNull());
    }

    /**
     * This {@link Instruction}s previous sibling instruction.
     *
     * @return The previous sibling {@link Instruction}  or <code>null</code>.
     * @see #getPrevSibling()
     */
    @Nullable
    public final Instruction getPrevSiblingOrNull() {
        if (inSlot == null || inSlot.prevSibling == null) return null;
        InstructionSlot<?> slot = inSlot.prevSibling.onPrevious(inSlot);
        return slot == null ? null : slot.getValueOrNull();
    }

    /**
     * This {@link Instruction}s previous sibling instruction.
     *
     * @return The previous sibling {@link Instruction}.
     * @see #getPrevSiblingOrNull()
     */
    public final Instruction getPrevSibling() {
        return requireNonNull(getPrevSiblingOrNull());
    }

    /**
     * This {@link Instruction}s first child instruction.
     *
     * @return The first child {@link Instruction}
     * or <code>null</code> if this instruction has no children.
     * @see #getFirstChild()
     */
    @Nullable
    public final Instruction getFirstChildOrNull() {
        if (firstChild == null) return null;
        InstructionSlot<?> slot = firstChild.onNext();
        return slot == null ? null : slot.getValueOrNull();
    }

    /**
     * This {@link Instruction}s first child instruction.
     *
     * @return The first child {@link Instruction}.
     * @see #getFirstChildOrNull()
     */
    public final Instruction getFirstChild() {
        return requireNonNull(getFirstChildOrNull());
    }

    /**
     * This {@link Instruction}s last child instruction.
     *
     * @return The last child {@link Instruction}
     * or <code>null</code> if this instruction has no children.
     * @see #getLastChild()
     */
    @Nullable
    public final Instruction getLastChildOrNull() {
        if (lastChild == null) return null;
        InstructionSlot<?> slot = lastChild.onPrevious(inSlot);
        return slot == null ? null : slot.getValueOrNull();
    }

    /**
     * This {@link Instruction}s last child instruction.
     *
     * @return The last child {@link Instruction}.
     * @see #getLastChildOrNull()
     */
    public final Instruction getLastChild() {
        return requireNonNull(getLastChildOrNull());
    }

    /**
     * Gets a {@link FastStream} containing this
     * {@link Instruction}s children.
     *
     * @return The {@link FastStream}.
     */
    public final FastStream<Instruction> getChildren() {
        InstructionSlot<?> firstChild = this.firstChild;
        if (firstChild == null) return FastStream.empty();

        return new FastStream<>() {
            @Override
            public Iterator<Instruction> iterator() {
                return new Iterator<>() {
                    @Nullable
                    Instruction next = getFirstChildOrNull();

                    @Override
                    public boolean hasNext() {
                        return next != null;
                    }

                    @Override
                    public Instruction next() {
                        if (next == null) throw new NoSuchElementException();

                        Instruction value = next;
                        next = next.getNextSiblingOrNull();
                        return value;
                    }
                };
            }

            @Override
            public void forEach(Consumer<? super Instruction> action) {
                Instruction next = getFirstChildOrNull();
                while (next != null) {
                    Instruction value = next;
                    next = next.getNextSiblingOrNull();
                    action.accept(value);
                }
            }
        };
    }

    /**
     * Gets a {@link FastStream} iterating all descendants
     * including this {@link Instruction} in post-order.
     *
     * @return The {@link FastStream}.
     */
    public final FastStream<Instruction> getDescendants() {
        Instruction firstChild = getFirstChildOrNull();
        if (firstChild == null) return of(this);

        return new FastStream<>() {

            @Override
            public Iterator<Instruction> iterator() {

                return new Iterator<>() {

                    @Nullable
                    private Instruction next = deepestChild(Instruction.this);

                    @Override
                    public boolean hasNext() {
                        return next != null;
                    }

                    @Override
                    public Instruction next() {
                        if (next == null) throw new NoSuchElementException();

                        Instruction ret = next;
                        next = nextInsn(next);
                        return ret;
                    }

                    private Instruction deepestChild(Instruction insn) {
                        while (true) {
                            Instruction child = insn.getFirstChildOrNull();
                            if (child == null) {
                                return insn;
                            }
                            insn = child;
                        }
                    }

                    @Nullable
                    private Instruction nextInsn(Instruction insn) {
                        if (insn == Instruction.this) return null;
                        Instruction next = insn.getNextSiblingOrNull();
                        if (next != null) return deepestChild(next);
                        return insn.getParent();
                    }
                };
            }

            @Override
            public void forEach(Consumer<? super Instruction> action) {
                forEachDescendent(action);
            }
        };
    }

    private void forEachDescendent(Consumer<? super Instruction> action) {
        Instruction next = getFirstChildOrNull();
        while (next != null) {
            Instruction value = next;
            next = next.getNextSiblingOrNull();
            value.forEachDescendent(action);
        }
        action.accept(this);
    }

    /**
     * Returns a filtered {@link FastStream} of descendants in post-order.
     *
     * @param type The Instruction class type to filter by.
     * @return The filtered {@link FastStream}.
     */
    public final <R extends Instruction> FastStream<R> descendantsOfType(Class<? extends R> type) {
        assert type.accessFlags().contains(AccessFlag.FINAL);
        return unsafeCast(getDescendants().filter(e -> e.getClass() == type));
    }

    /**
     * Returns a filtered {@link FastStream} of descendants in post-order.
     * <p>
     * This function is intended to be used by static matching methods.
     *
     * @param filter The filter to apply.
     * @return The filtered {@link FastStream}.
     */
    public final <R extends Instruction> FastStream<R> descendantsWhere(Predicate<Instruction> filter) {
        return unsafeCast(getDescendants().filter(filter));
    }

    /**
     * Returns a filtered {@link FastStream} of descendants in post-order.
     * <p>
     * This function is intended to be used by static matching methods which return
     * <code>null</code> to indicate a failed match.
     *
     * @param filter The filter to apply.
     * @return The filtered {@link FastStream}.
     */
    public final <R extends Instruction> FastStream<R> descendantsMatching(Function<Instruction, @Nullable R> filter) {
        return getDescendants().map(filter).filter(Objects::nonNull);
    }

    /**
     * Returns a {@link LinkedList} of descendants in post-order.
     *
     * @param type The Instruction class type to filter by.
     * @return The {@link LinkedList}.
     */
    public final <R extends Instruction> LinkedList<R> descendantsToList(Class<? extends R> type) {
        return this.<R>descendantsOfType(type).toLinkedList();
    }

    /**
     * Returns a {@link LinkedList} of descendants in post-order.
     * <p>
     * This function is intended to be used by static matching methods.
     *
     * @param filter The filter to apply.
     * @return The {@link LinkedList}.
     */
    public final <R extends Instruction> LinkedList<R> descendantsToListWhere(Predicate<Instruction> filter) {
        return this.<R>descendantsWhere(filter).toLinkedList();
    }

    /**
     * Returns the first parent {@link Instruction} with the provided type.
     *
     * @param type The Instruction class type to filter by.
     * @return The first parent {@link Instruction}.
     */
    @SuppressWarnings ("unchecked")
    public final <R extends Instruction> R firstAncestorOfType(Class<? extends R> type) {
        assert type.accessFlags().contains(AccessFlag.FINAL);
        Instruction insn = this;
        while (insn.getClass() != type) {
            insn = insn.getParent();
        }
        return (R) insn;
    }

    /**
     * Returns a filtered {@link FastStream} of ancestors.
     *
     * @param type The Instruction class type to filter by.
     * @return The filtered {@link FastStream}.
     */
    public final <R extends Instruction> FastStream<R> ancestorsOfType(Class<? extends R> type) {
        return new FastStream<>() {
            @NotNull
            @Override
            public Iterator<R> iterator() {
                return new Iterator<>() {

                    @Nullable
                    private R next = firstAncestorOfTypeOrDefault(Instruction.this);

                    @Override
                    public boolean hasNext() {
                        return next != null;
                    }

                    @Override
                    public R next() {
                        if (next == null) throw new NoSuchElementException();

                        R insn = next;
                        next = firstAncestorOfTypeOrDefault(insn.getParentOrNull());
                        return insn;
                    }

                    @Nullable
                    @SuppressWarnings ("unchecked")
                    private R firstAncestorOfTypeOrDefault(@Nullable Instruction insn) {
                        while (insn != null && insn.getClass() != type) {
                            insn = insn.getParentOrNull();
                        }
                        return (R) insn;
                    }
                };
            }

            @Override
            @SuppressWarnings ("unchecked")
            public void forEach(Consumer<? super R> action) {
                Instruction insn = Instruction.this;
                while (insn != null) {
                    if (insn.getClass() == type) {
                        action.accept((R) insn);
                    }
                    insn = insn.getParentOrNull();
                }
            }
        };
    }

    /**
     * Checks if this {@link Instruction} or one of its parents
     * is <code>possibleAncestor</code>.
     *
     * @param possibleAncestor The ancestor to check against.
     * @return If this {@link Instruction} is a descendant of <code>possibleAncestor</code>.
     */
    public final boolean isDescendantOf(@Nullable Instruction possibleAncestor) {
        if (possibleAncestor == null) return false;
        Instruction candidate = this;
        while (candidate != null) {
            if (candidate == possibleAncestor) {
                return true;
            }
            candidate = candidate.getParentOrNull();
        }

        return false;
    }

    /**
     * Increments the reference count of this instruction.
     * <p>
     * This should generally not be directly used.
     *
     * @see #releaseRef()
     * // TODO, document what a 'reference' is
     */
    public final void addRef() {
        if (refCount == -1) { throw new IllegalStateException("Attempted to revive dead instruction."); }
        if (refCount++ == 0) {
            onConnected();
        }
    }

    /**
     * Decrements the reference count of this instruction.
     * <p>
     * This should generally not be directly used.
     *
     * @see #addRef()
     */
    public final void releaseRef() {
        assert refCount > 0;
        if (--refCount == 0) {
            refCount = -1;
            inSlot = null;
            onDisconnected();
        }
    }

    /**
     * Used to check if this {@link Instruction} has any references
     * and is connected to a tree.
     *
     * @return If this {@link Instruction} is connected.
     */
    public final boolean isConnected() {
        return refCount > 0;
    }

    /**
     * Removes this {@link Instruction} from an {@link InstructionCollection}.
     *
     * @throws UnsupportedOperationException If this {@link Instruction} is not contained within a collection.
     */
    public final void remove() {
        assert inSlot != null;
        if (!inSlot.isInCollection()) {
            throw new UnsupportedOperationException("Instruction is not inside a collection.");
        }
        inSlot.remove();
        inSlot = null;
    }

    /**
     * Inserts an {@link Instruction} before this one inside an {@link InstructionCollection}.
     *
     * @param value The {@link Instruction} to insert.
     * @throws UnsupportedOperationException If this {@link Instruction} is not contain
     */
    public final void insertBefore(Instruction value) {
        if (inSlot != null) {
            inSlot.insertBefore(value);
        }
    }

    /**
     * Inserts an {@link Instruction} after this one inside an {@link InstructionCollection}.
     *
     * @param value The {@link Instruction} to insert.
     * @throws UnsupportedOperationException If this {@link Instruction} is not contain
     */
    public final void insertAfter(Instruction value) {
        if (inSlot != null) {
            inSlot.insertAfter(value);
        }
    }

    /**
     * Replace this {@link Instruction} in the tree with another {@link Instruction}.
     * <p>
     * This function will efficiently remove and unlink the instruction from
     * wherever else it was in the tree prior if required.
     *
     * @param value The value to replace with.
     */
    public final <T extends Instruction> T replaceWith(T value) {
        if (value == this) return value;

        assert inSlot != null;
        inSlot.set(value);
        return value;
    }

    /**
     * An optional data tag for this instruction.
     *
     * @return The tag.
     */
    @Nullable
    public final InsnTag getTag() {
        return tag;
    }

    /**
     * Sets the optional data tag for this instruction.
     *
     * @param tag The data tag. <code>null</code> to clear.s
     * @throws IllegalStateException If this {@link Instruction} already has a tag set.
     */
    public final void setTag(@Nullable InsnTag tag) {
        if (this.tag != null && tag != null) throw new IllegalStateException("Can't replace tag.");
        this.tag = tag;
    }

    /**
     * Gets the offset of this instruction in bytecode.
     *
     * @return The offset.
     */
    public final int getBytecodeOffset() {
        return bytecodeOffset;
    }

    /**
     * Set the offset for this instruction in bytecode.
     *
     * @param bytecodeOffset The offset.
     */
    public final void setBytecodeOffset(int bytecodeOffset) {
        this.bytecodeOffset = bytecodeOffset;
    }

    /**
     * Sets the bytecode offset and tracked source line of this instruction
     * to that of the given instruction.
     *
     * @param sourceInsn The source instruction to copy from.
     * @return This instruction.
     */
    public final <T extends Instruction> T withOffsets(Instruction sourceInsn) {
        setOffsets(sourceInsn);
        return unsafeCast(this);
    }

    /**
     * Sets the bytecode offset and tracked source line of this instruction
     * to that of the given instruction.
     *
     * @param sourceInsn The source instruction to copy from.
     */
    public final void setOffsets(Instruction sourceInsn) {
        setBytecodeOffset(sourceInsn.getBytecodeOffset());
        setSourceLine(sourceInsn.getSourceLine());
    }

    /**
     * Gets the captured source line as represented by the LineNumberTable.
     *
     * @return The line number.
     */
    public final int getSourceLine() {
        return sourceLine;
    }

    /**
     * Sets the captured source line.
     *
     * @param sourceLine The source line.
     */
    public final void setSourceLine(int sourceLine) {
        this.sourceLine = sourceLine;
    }

    @Override
    public String toString() {
        return toString(DebugPrintOptions.DEFAULT);
    }

    public String toString(DebugPrintOptions opts) {
        AstSourceVisitor visitor = new AstSourceVisitor(opts);

        LineBuffer ast = accept(visitor);

        // Try grabbing the top most class decl if it exists in the tree, cleans up the imports a bit.
        Instruction last = this;
        while (last.getParentOrNull() != null) {
            last = last.getParentOrNull();
        }

        // TODO, Re-evaluate how imports should be printed in toString mode.
        if (true) return ast.toString();
        List<String> imports = of(visitor.getImports(last instanceof ClassDecl ? (ClassDecl) last : null))
                .map(e -> "import " + e)
                .toImmutableList();
        if (imports.isEmpty()) return ast.toString();
        return LineBuffer.of(imports)
                .add("")
                .join(ast)
                .toString();
    }

    protected void invalidateFlags() {
        Instruction insn = this;
        while (insn != null && insn.flags != InstructionFlag.INVALID_FLAGS) {
            insn.flags = InstructionFlag.INVALID_FLAGS;
            insn = insn.getParentOrNull();
        }
    }

    @MustBeInvokedByOverriders
    protected void onConnected() {
        InstructionSlot<?> slot = firstChild;
        while (slot != null) {
            slot.onConnected();
            slot = slot.nextSibling;
        }
    }

    @MustBeInvokedByOverriders
    protected void onDisconnected() {
        InstructionSlot<?> next = firstChild;

        // Special iteration to skip slots which have had their value removed.
        while (next != null && (next = next.onNext()) != null) {
            // Explicitly use value directly as we do need to iterate raw tree state here.
            Instruction value = next.value;
            if (value != null) {
                value.releaseRef();
            }

            next = next.nextSibling;
        }
    }

    protected void onChildModified() {
    }

    /**
     * Called to compute the exposed flags exposed via {@link Instruction#getFlags()}.
     *
     * @return The computed flags.
     */
    protected EnumBitSet<InstructionFlag> computeFlags() {
        EnumBitSet<InstructionFlag> flags = getDirectFlags().clone();
        getChildren().forEach(e -> flags.or(e.getFlags()));
        return flags;
    }
}
