package net.covers1624.coffeegrinder.source;

import net.covers1624.coffeegrinder.bytecode.*;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import net.covers1624.coffeegrinder.bytecode.insns.tags.CompactConstructorTag;
import net.covers1624.coffeegrinder.bytecode.insns.tags.ErrorTag;
import net.covers1624.coffeegrinder.bytecode.matching.BranchLeaveMatching;
import net.covers1624.coffeegrinder.bytecode.matching.InvokeMatching;
import net.covers1624.coffeegrinder.bytecode.transform.transformers.NumericConstants;
import net.covers1624.coffeegrinder.type.*;
import net.covers1624.coffeegrinder.type.AnnotationSupplier.TypeAnnotationLocation;
import net.covers1624.coffeegrinder.util.EnumBitSet;
import net.covers1624.coffeegrinder.util.None;
import net.covers1624.quack.collection.FastStream;
import org.jetbrains.annotations.Nullable;

import java.util.*;

import static net.covers1624.coffeegrinder.bytecode.insns.Invoke.InvokeKind;
import static net.covers1624.coffeegrinder.bytecode.matching.AssignmentMatching.matchPreInc;
import static net.covers1624.coffeegrinder.bytecode.matching.LdcMatching.matchNegation;
import static net.covers1624.coffeegrinder.source.LineBuffer.of;
import static net.covers1624.coffeegrinder.source.LineBuffer.paren;
import static net.covers1624.quack.util.SneakyUtils.notPossible;

/**
 * Created by covers1624 on 19/7/21.
 */
public class JavaSourceVisitor extends AbstractSourceVisitor {

    private final TypeResolver typeResolver;
    private final boolean showImplicits;
    private final NumericConstants numericConstants;
    private final LinkedList<ClassDecl> classStack = new LinkedList<>();
    private final LinkedList<MethodDecl> functionStack = new LinkedList<>();
    private final Map<LocalVariable, String> variableNames = new HashMap<>();
    private final Set<LocalVariable> declaredVariables = new HashSet<>();
    private final Set<Reference> localRefDeclarations = new HashSet<>();
    private final Set<Field> declaredFields = new HashSet<>();

    private final ScopeVisitor<LineBuffer, None> scopeVisitor = new ScopeVisitor<>(this);

    public JavaSourceVisitor(TypeResolver typeResolver) {
        this(typeResolver, false);
    }

    public JavaSourceVisitor(TypeResolver typeResolver, boolean showImplicits) {
        super(typeResolver);
        this.typeResolver = typeResolver;
        this.showImplicits = showImplicits;
        numericConstants = new NumericConstants(typeResolver);
        importCollector.setConstantPrinter(num -> {
            Instruction unfolded = numericConstants.getReplacement(num);
            if (unfolded != null) {
                unfolded.addRef(); // connect the unfolded hallucination so queries on parents in binary operations and such work.
                return lines(unfolded);
            }
            return lines(new LdcNumber(num));
        });
        importCollector.setIsVariableDeclared(scopeVisitor::isDeclared);
    }

    @Override
    public LineBuffer lines(Instruction other) {
        return scopeVisitor.visit(other, None.INSTANCE);
    }

    private LineBuffer end(LineBuffer buffer, boolean end) {
        if (!end) return buffer;
        if (buffer.lines.isEmpty()) return buffer;
        return buffer.append(";");
    }

    private boolean shouldEnd(Instruction insn) {
        return switch (insn) {
            case DeadCode ignored -> false;
            case Block ignored -> false;
            case BlockContainer ignored -> false;
            case ClassDecl ignored -> false;
            case ForEachLoop ignored -> false;
            case ForLoop ignored -> false;
            case IfInstruction ignored -> false;
            case Switch ignored -> false;
            case SwitchTable ignored -> false;
            case Synchronized ignored -> false;
            case TryCatch ignored -> false;
            case TryFinally ignored -> false;
            case TryWithResources ignored -> false;
            case WhileLoop ignored -> false;
            default -> true;
        };
    }

    private LineBuffer linesWrap(Instruction insn, OperatorPrecedence precedence) {
        return wrap(lines(insn), getPrecedence(insn).isLowerThan(precedence));
    }

    // TODO, add a `braced` method similar to this, pushes scope and adds curly braces
    private LineBuffer wrap(LineBuffer buffer, boolean wrap) {
        if (!wrap) return buffer;
        return buffer.prepend("(").append(")");
    }

    private BlockContainer getOwningContainer(Instruction insn) {
        while (!(insn.getParent() instanceof BlockContainer)) {
            insn = insn.getParent();
        }
        return (BlockContainer) insn.getParent();
    }

    private AbstractLoop getOwningLoop(Instruction insn) {
        while (!(insn.getParent() instanceof AbstractLoop)) {
            insn = insn.getParent();
        }
        return (AbstractLoop) insn.getParent();
    }

    @Nullable
    private Instruction getNaturalExit(BlockContainer container) {
        if (showImplicits) return null;

        Instruction lastInsn = container.blocks.last().instructions.last();
        return switch (lastInsn) {
            case Leave leave when leave.getTargetContainer() == container && !(container.getParent() instanceof AbstractLoop) -> lastInsn;
            case Continue aContinue when aContinue.getLoop().getBody() == container -> lastInsn;
            case Return aReturn when aReturn.getMethod().getBody() == container -> lastInsn;
            default -> null;
        };
    }

    private OperatorPrecedence getPrecedence(Instruction insn) {
        return switch (insn) {
            case Binary binary when matchNegation(binary) != null -> OperatorPrecedence.UNARY;
            case Binary binary -> switch (binary.getOp()) {
                case SUB, ADD -> OperatorPrecedence.ADDITIVE;
                case MUL, DIV, REM -> OperatorPrecedence.MULTIPLICATIVE;
                case AND -> OperatorPrecedence.BITWISE_AND;
                case OR -> OperatorPrecedence.BITWISE_INCLUSIVE_OR;
                case XOR -> OperatorPrecedence.BITWISE_EXCLUSIVE_OR;
                case SHIFT_LEFT, SHIFT_RIGHT, LOGICAL_SHIFT_RIGHT -> OperatorPrecedence.SHIFT;
            };
            case LogicNot logicNot -> OperatorPrecedence.UNARY;
            case Switch swtch -> OperatorPrecedence.UNARY;
            case Comparison comparison -> switch (comparison.getKind()) {
                case EQUAL, NOT_EQUAL -> OperatorPrecedence.EQUALITY;
                case GREATER_THAN, GREATER_THAN_EQUAL, LESS_THAN, LESS_THAN_EQUAL -> OperatorPrecedence.RELATIONAL;
            };
            case InstanceOf instanceOf -> OperatorPrecedence.RELATIONAL;
            case LogicAnd logicAnd -> OperatorPrecedence.LOGIC_AND;
            case LogicOr logicOr -> OperatorPrecedence.LOGIC_OR;
            case Ternary ternary -> OperatorPrecedence.TERNARY;
            case Cast cast -> OperatorPrecedence.CAST;
            case FieldReference fieldRef -> OperatorPrecedence.MEMBER_ACCESS;
            case Invoke invoke -> OperatorPrecedence.MEMBER_ACCESS;
            case InvokeDynamic indy -> OperatorPrecedence.MEMBER_ACCESS;
            case ArrayLen arrayLen -> OperatorPrecedence.MEMBER_ACCESS;
            case ArrayElementReference aer -> OperatorPrecedence.MEMBER_ACCESS;
            // Semantics of instruction are postfix if inside a block, as we flip it to a post increment for readability.
            case CompoundAssignment compoundAssignment when matchPreInc(insn) != null -> insn.getParent() instanceof Block ? OperatorPrecedence.POSTFIX_INC_DEC : OperatorPrecedence.UNARY;
            case CompoundAssignment compoundAssignment -> OperatorPrecedence.ASSIGNMENT;
            case Store store -> OperatorPrecedence.ASSIGNMENT;
            case PostIncrement postIncrement -> OperatorPrecedence.POSTFIX_INC_DEC;
            default -> OperatorPrecedence.UNKNOWN;
        };
    }

    public String getVariableName(LocalVariable v) {
        return variableNames.computeIfAbsent(v, e-> {
            var name = v.getName();
            if (v.getKind() != LocalVariable.VariableKind.PARAMETER && scopeVisitor.isDeclared(name)) {
                name = v.getUniqueName();
                if (name.equals(v.getName()))
                    name += "$0";
            }
            return name;
        });
    }

    public boolean requiresLabelDefinition(Leave leave) {
        if (!supportsNaturalBreakKeyword(leave.getTargetContainer())) return true;

        MethodDecl func = peekFunc();
        if (leave.getTargetContainer() == func.getBody()) return false;

        // Continually iterate up the BlockContainer chain.
        BlockContainer container = getOwningContainer(leave);
        while (true) {
            // If we reach the target, we don't need a label
            if (leave.getTargetContainer() == container) return false;
            // If the container we found is either a switch or a loop, we need a label to leave through it.
            if (supportsNaturalBreakKeyword(container)) return true;
            container = getOwningContainer(container);
        }
    }

    private boolean supportsNaturalBreakKeyword(BlockContainer container) {
        return container.getParent() instanceof Switch || container.getParent() instanceof AbstractLoop;
    }

    public boolean requiresLabelDefinition(Continue cont) {
        return getOwningLoop(cont) != cont.getLoop();
    }

    public LineBuffer blockHeader(BlockContainer container) {
        if (requiresLabelDefinition(container)) {
            return of(container.getEntryPoint().getName() + ":");
        }
        return of();
    }

    public boolean requiresLabelDefinition(BlockContainer container) {
        //Might need some extra logic to support the fact that a 'fallthrough exit' from a try isn't actually printed
        if (!supportsNaturalBreakKeyword(container)
            && container.getLeaveCount() == 1
            && isFallthroughExit(container.getLeaves().only())) {
            return false;
        }

        if (container.getParent() instanceof AbstractLoop) {
            AbstractLoop loop = (AbstractLoop) container.getParent();
            if (loop.getContinues().anyMatch(this::requiresLabelDefinition)) {
                return true;
            }
        }
        return container.getLeaves().anyMatch(this::requiresLabelDefinition);
    }

    private boolean isFallthroughExit(Leave leave) {
        return leave.getTargetContainer().blocks.last().getLastChild() == leave;
    }

    private MethodDecl peekFunc() {
        return Objects.requireNonNull(functionStack.peek());
    }

    private @Nullable ClassDecl peekClass() {
        return classStack.peek();
    }

    @Override
    public LineBuffer visitDefault(Instruction insn, None ctx) {
        throw new UnsupportedOperationException("Unhandled visit method: " + insn.getClass().getName());
    }

    @Override
    public LineBuffer visitNop(Nop nop, None ctx) {
        if (nop.getTag() instanceof ErrorTag(String comment, String literal)) {
            return of("/*").append(comment).append("*/ ").append(literal);
        }
        return of("nop");
    }

    @Override
    public LineBuffer visitDeadCode(DeadCode deadCode, None ctx) {
        return of("/* Dead Code")
                .join(lines(deadCode.getCode()))
                .append("*/");
    }

    @Override
    public LineBuffer visitArrayElementReference(ArrayElementReference elemRef, None ctx) {
        return linesWrap(elemRef.getArray(), OperatorPrecedence.MEMBER_ACCESS)
                .append("[")
                .append(lines(elemRef.getIndex()))
                .append("]");
    }

    @Override
    public LineBuffer visitArrayLen(ArrayLen arrayLen, None ctx) {
        return linesWrap(arrayLen.getArray(), OperatorPrecedence.MEMBER_ACCESS).append(".length");
    }

    @Override
    public LineBuffer visitAssert(Assert assertInsn, None ctx) {
        LineBuffer buffer = of("assert ")
                .append(lines(assertInsn.getCondition()));
        Instruction message = assertInsn.getMessage();
        if (!(message instanceof Nop)) {
            buffer = buffer.append(" : ")
                    .append(lines(message));
        }
        return buffer;
    }

    @Override
    public LineBuffer visitBlock(Block block, None ctx) {
        // TODO, fori use a Block for the increment but its actually a comma seperated list of statements.
        if (block.getParent() instanceof ForLoop loop && loop.getIncrement() == block) {
            return argList("", block.instructions, "");
        }

        List<String> builder = new ArrayList<>();
        boolean emitBraces = false;
        // Emit header and braces if we have incoming branches, or we are in a container and have no incoming edges (early ast mid transforms).
        if (showImplicits || !block.getBranches().isEmpty() || block.getParent() instanceof BlockContainer && block.getIncomingEdgeCount() != 1) {
            builder.add(indent(block.getName() + ":"));
            emitBraces = true;
        }

        if (emitBraces) {
            builder.add(indent("{"));
            pushIndent();
        }
        for (Instruction instruction : block.instructions) {
            builder.addAll(indent(end(lines(instruction), shouldEnd(instruction))).lines);
        }
        if (emitBraces) {
            popIndent();
            builder.add(indent("}"));
        }
        return of(builder);
    }

    @Override
    public LineBuffer visitBlockContainer(BlockContainer container, None ctx) {
        boolean forGoto = container.getParentOrNull() instanceof Block;
        boolean requiresBraces = container.blocks.size() != 1        // Can't emit raw label for multiple blocks.
                                 || container.blocks.first().instructions.size() > 2  // Can't emit raw label for more than 2 instructions.
                                 || BranchLeaveMatching.matchLeave(container.blocks.first().getLastChild(), container) == null; // Can't emit raw label if the last (of the 2) instruction is not a leave from the container.
        LineBuffer buffer = of();
        if (forGoto) {
            buffer = blockHeader(container);
            if (requiresBraces) {
                buffer = buffer.join(of("{"));
                pushIndent();
            }
        }

        boolean first = true;
        for (Block block : container.blocks) {
            if (!first) {
                buffer = buffer.add("");
            }
            first = false;
            buffer = buffer.join(lines(block));
        }

        if (forGoto && requiresBraces) {
            popIndent();
            buffer = buffer.join(of("}"));
        }
        return buffer;
    }

    @Override
    public LineBuffer visitBranch(Branch branch, None ctx) {
        Block targetBlock = branch.getTargetBlock();
        return of("goto " + targetBlock.getName());
    }

    @Override
    public LineBuffer visitCheckCast(Cast cast, None ctx) {
        return of("(" + importCollector.collect(cast.getType()) + ") ").append(linesWrap(cast.getArgument(), OperatorPrecedence.CAST));
    }

    private LineBuffer classHeader(ClassDecl classDecl, boolean isEnum) {
        ClassType clazz = classDecl.getClazz();
        EnumBitSet<AccessFlag> classFlags = clazz.getAccessFlags();
        boolean isAnnotation = classFlags.get(AccessFlag.ANNOTATION);
        boolean isInterface = classFlags.get(AccessFlag.INTERFACE) && !isAnnotation;
        boolean isRecord = classFlags.get(AccessFlag.RECORD);

        LineBuffer buffer = of();
        if (clazz.getDeclType().isLocalOrAnonymous() && classDecl.getParentOrNull() != null && classDecl.getParent() instanceof ClassDecl) {
            // If the class is a Local/Anon class that has not been moved yet.
            buffer = buffer.add("// CG: Unprocessed Local/Anonymous Class");
        }

        AnnotationSupplier annotationSupplier = clazz.getAnnotationSupplier();

        buffer = buffer.join(importCollector.annotations(annotationSupplier.getAnnotations()));

        // Annoyingly, we need to insert 'sealed' and 'non-sealed' before the 'interface' access flag.
        if (isInterface) {
            classFlags = classFlags.copy();
            classFlags.clear(AccessFlag.INTERFACE);
            classFlags.clear(AccessFlag.ABSTRACT);
        }

        // Records are always considered static (in local/inner context) and final.
        if (isRecord) {
            classFlags = classFlags.copy();
            classFlags.clear(AccessFlag.STATIC);
            classFlags.clear(AccessFlag.FINAL);
        }

        // Local enums are always declared static, and can't declare the modifier.
        if (isEnum && clazz.getDeclType() == ClassType.DeclType.LOCAL) {
            classFlags = classFlags.copy();
            classFlags.clear(AccessFlag.STATIC);
        }

        buffer = buffer.add(AccessFlag.stringRep(classFlags));
        List<ClassType> permittedSubclasses = clazz.getPermittedSubclasses();

        // Enums, annotations, and records don't have sealed classes or 'interface' and 'class' modifiers.
        if (!isEnum && !isAnnotation && !isRecord) {
            if (!permittedSubclasses.isEmpty()) {
                buffer = buffer.append("sealed ");
            } else if (clazz.getDirectSuperTypes().anyMatch(e -> !e.getPermittedSubclasses().isEmpty()) && !clazz.isFinal()) {
                buffer = buffer.append("non-sealed ");
            }

            if (isInterface) {
                buffer = buffer.append("interface ");
            } else {
                buffer = buffer.append("class ");
            }
        }

        buffer = buffer.append(clazz.getName());
        buffer = buffer.append(typeParameters(clazz, annotationSupplier));

        // Emit record members.
        if (isRecord) {
            // Push temporary scope so record fields get the same print semantics as regular params.
            classStack.push(classDecl);
            importCollector.pushScope(clazz);
            buffer = buffer.append("(");
            boolean first = true;
            for (ClassDecl.RecordComponentDecl comp : classDecl.recordComponents) {
                if (!first) {
                    buffer = buffer.append(", ");
                }
                first = false;

                if (!comp.regularAnnotations.isEmpty()) {
                    buffer = buffer.append(importCollector.annotations(comp.regularAnnotations).joinOn(" "))
                            .append(" ");
                }
                AType type = comp.field.getField().getType();
                if (comp.isVarargs) {
                    assert type instanceof ArrayType;
                    assert classDecl.recordComponents.getLast() == comp;
                    type = ((ArrayType) type).getElementType();
                }
                buffer = buffer.append(importCollector.collect(type, comp.typeAnnotations));
                if (comp.isVarargs) {
                    buffer = buffer.append("...");
                }

                buffer = buffer.append(" ")
                        .append(comp.field.getField().getName());
            }
            buffer = buffer.append(")");
            classStack.pop();
            importCollector.popScope();
        }

        buffer = buffer.append(' ');

        // We can ignore the superclass for Enums, interfaces and Object.
        ClassType superClass = clazz.getSuperClass();
        TypeAnnotationData superAnnotations = annotationSupplier.getTypeAnnotations(TypeAnnotationLocation.CLASS_EXTENDS, superClass);
        if (!isEnum && !isRecord && !clazz.isInterface() && !TypeSystem.isObject(superClass) || !superAnnotations.isEmpty()) {
            buffer = buffer.append("extends ").append(importCollector.collect(superClass, superAnnotations)).append(" ");
        }

        // We can ignore any interfaces for annotations.
        List<ClassType> interfaces = clazz.getInterfaces();
        if (!isAnnotation && !interfaces.isEmpty()) {
            buffer = buffer.append(clazz.isInterface() ? "extends " : "implements ");
            for (int i = 0; i < interfaces.size(); i++) {
                if (i != 0) {
                    buffer = buffer.append(", ");
                }
                ClassType type = interfaces.get(i);
                TypeAnnotationData annotation = annotationSupplier.getTypeAnnotations(TypeAnnotationLocation.CLASS_INTERFACE, type, i);
                buffer = buffer.append(importCollector.collect(type, annotation));
            }
            buffer = buffer.append(" ");
        }

        // Ditto, as above, enums use 'permittedSubclasses' for implementations.
        if (!isEnum && !permittedSubclasses.isEmpty()) {
            buffer = buffer.append("permits ");
            for (int i = 0; i < permittedSubclasses.size(); i++) {
                if (i != 0) {
                    buffer = buffer.append(", ");
                }
                buffer = buffer.append(importCollector.collect(permittedSubclasses.get(i)));
            }
            buffer = buffer.append(" ");
        }

        return buffer;
    }

    private LineBuffer visitPackageInfo(ClassDecl decl) {
        ClassType clazz = decl.getClazz();
        classStack.push(decl);
        importCollector.pushScope(clazz);
        LineBuffer buffer = importCollector.annotations(clazz.getAnnotationSupplier().getAnnotations());
        buffer = buffer.add("package " + clazz.getPackage() + ";");
        buffer = buffer.add("");
        List<String> imports = importCollector.getImports(decl);
        if (!imports.isEmpty()) {
            for (String s : imports) {
                buffer = buffer.add("import " + s + ";");
            }
        }
        importCollector.popScope();
        classStack.pop();
        return buffer;
    }

    @Override
    public LineBuffer visitClassDecl(ClassDecl classDecl, None ctx) {
        ClassType clazz = classDecl.getClazz();
        boolean isTopLevel = classStack.isEmpty();
        boolean isAnonymousClass = classDecl.getParentOrNull() instanceof New;
        if (isTopLevel) {
            importCollector.setTopLevelClass(clazz);
        }

        if (clazz.isInterface() && clazz.isSynthetic() && clazz.getName().equals("package-info")) {
            return visitPackageInfo(classDecl);
        }

        boolean isEnum = clazz.isEnum() && clazz.getDeclType() != ClassType.DeclType.ANONYMOUS;
        boolean isRecord = clazz.getAccessFlags().get(AccessFlag.RECORD);

        LineBuffer buffer;
        if (isAnonymousClass) {
            buffer = of("{");
        } else {
            buffer = indent(classHeader(classDecl, isEnum).append("{"));
        }
        classStack.push(classDecl);
        importCollector.pushScope(clazz);
        pushIndent();

        List<FieldDecl> fields = classDecl.getFieldMembers().toImmutableList();
        if (isEnum) {
            int processedEnums = 0;
            for (FieldDecl decl : fields) {
                if (!decl.getField().getAccessFlags().get(AccessFlag.ENUM)) break;

                if (processedEnums != 0) {
                    buffer = buffer.append(",");
                }
                buffer = buffer.join(indent(visitEnumFieldDecl(decl)));

                processedEnums++;
            }
            if (processedEnums != 0) {
                // End the enums on the existing line.
                buffer = buffer.append(";");
            } else if (!classDecl.members.isEmpty()) {
                // Add newline with semi-colon to end enums block, only if we have members, otherwise its redundant.
                buffer = buffer.add(indent(";"));
            }
            fields = fields.subList(processedEnums, fields.size());
        }

        boolean first = true;
        for (FieldDecl field : fields) {
            if (isRecord) {
                if (!field.getField().isStatic()) {
                    continue;
                }
            }
            if (first) {
                // Add spacer
                buffer = buffer.add("");
                first = false;
            }
            buffer = buffer.join(indent(lines(field)));
        }

        for (MethodDecl method : classDecl.getMethodMembers().toImmutableList()) {
            buffer = buffer.add("");
            buffer = buffer.join(indent(lines(method)));
        }

        for (ClassDecl inner : classDecl.getClassMembers().toImmutableList()) {
            buffer = buffer.add("");
            buffer = buffer.join(indent(lines(inner)));
        }
        popIndent();
        buffer = buffer.add(indent("}"));
        importCollector.popScope();
        classStack.pop();

        // Build class 'top' for top-level classes.
        LineBuffer classTop = of();
        if (isTopLevel) {
            if (!clazz.getPackage().isEmpty()) {
                classTop = classTop.add("package " + clazz.getPackage() + ";");
                classTop = classTop.add("");
            }
            List<String> imports = importCollector.getImports(classDecl);
            if (!imports.isEmpty()) {
                for (String s : imports) {
                    classTop = classTop.add("import " + s + ";");
                }
                classTop = classTop.add("");
            }
        }

        return classTop.join(buffer);
    }

    @Override
    public LineBuffer visitCompare(Compare compare, None ctx) {
        var left = compare.getLeft();
        var right = compare.getRight();
        return of("__compare.").append(compare.getKind().name().toLowerCase())
                .append("(")
                .append(wrap(lines(left), getPrecedence(left).isLowerThan(OperatorPrecedence.RELATIONAL)))
                .append(" " + compare.getKind().getChar() + " ")
                .append(wrap(lines(right), !OperatorPrecedence.RELATIONAL.isLowerThan(getPrecedence(right))))
                .append(")");
    }

    @Override
    public LineBuffer visitComparison(Comparison comparison, None ctx) {
        boolean wrapLeft = getPrecedence(comparison.getLeft()).isLowerThan(getPrecedence(comparison));
        boolean wrapRight = !getPrecedence(comparison).isLowerThan(getPrecedence(comparison.getRight()));
        return wrap(lines(comparison.getLeft()), wrapLeft).append(" " + comparison.getKind().chars + " ").append(wrap(lines(comparison.getRight()), wrapRight));
    }

    @Override
    public LineBuffer visitCompoundAssignment(CompoundAssignment comp, None ctx) {
        // Handle pre increment/decrement
        // i += 1 -> ++i
        // i -= 1 -> --i
        if (matchPreInc(comp) != null) {
            String op = comp.getOp() == BinaryOp.ADD ? "++" : "--";

            // Flip pre-increment to post-increment if its inside a block.
            // ++i -> i++
            // --i -> i--
            if (comp.getParent() instanceof Block) {
                return lines(comp.getReference()).append(op);
            } else {
                return of(op).append(lines(comp.getReference()));
            }
        }

        return lines(comp.getReference())
                .append(" " + comp.getOp().chars + "= ")
                .append(lines(comp.getValue()));
    }

    @Override
    public LineBuffer visitContinue(Continue cont, None ctx) {
        // Don't print the natural exit.
        if (getNaturalExit(getOwningContainer(cont)) == cont) {
            return of();
        }

        if (!requiresLabelDefinition(cont)) {
            return of("continue");
        }
        return of("continue " + cont.getLoop().getBody().getEntryPoint().getName());
    }

    @Override
    public LineBuffer visitDoWhileLoop(DoWhileLoop doWhileLoop, None ctx) {
        LineBuffer buffer = blockHeader(doWhileLoop.getBody());
        buffer = buffer.add("do {");
        pushIndent();
        buffer = buffer.join(indent(lines(doWhileLoop.getBody())));
        popIndent();
        buffer = buffer.add("}");
        buffer = buffer.add("while (").append(lines(doWhileLoop.getCondition())).append(")");
        return buffer;
    }

    private LineBuffer visitEnumFieldDecl(FieldDecl fieldDecl) {
        // Pre-field initializers
        if ((fieldDecl.getValue() instanceof Nop)) return of();

        New newInsn = (New) fieldDecl.getValue();
        Field field = fieldDecl.getField();
        LineBuffer buffer = of();
        List<AnnotationData> annotations = field.getAnnotationSupplier().getAnnotations();
        if (!annotations.isEmpty()) {
            buffer = buffer.join(importCollector.annotations(annotations));
            buffer = buffer.add("");
        }
        buffer = buffer.append(field.getName());

        if (newInsn.getArguments().anyMatch(e -> !(e instanceof Nop))) {
            buffer = buffer.append(argList(newInsn.getArguments()));
        }

        if (newInsn.hasAnonymousClassDeclaration()) {
            buffer = buffer.append(" ").append(lines(newInsn.getAnonymousClassDeclaration()));
        }
        // After the child has been evaluated.
        declaredFields.add(field.getDeclaration());
        return buffer;
    }

    @Override
    public LineBuffer visitFieldDecl(FieldDecl fieldDecl, None ctx) {
        Field field = fieldDecl.getField();
        String access = AccessFlag.stringRep(field.getAccessFlags());
        if (fieldDecl.getField().getDeclaringClass().isInterface()) {
            // Only valid modifiers for fields in interfaces are public static final.
            // These are implicit and cannot be removed
            access = "";
        }
        AType fieldType = field.getType();
        AnnotationSupplier annotationSupplier = field.getAnnotationSupplier();
        TypeAnnotationData typeAnnotations = annotationSupplier.getTypeAnnotations(TypeAnnotationLocation.FIELD, fieldType);
        LineBuffer buffer = importCollector.annotations(annotationSupplier.getAnnotations(typeAnnotations))
                .add(access)
                .append(importCollector.collect(fieldType, typeAnnotations))
                .append(" ")
                .append(field.getName());
        if (!(fieldDecl.getValue() instanceof Nop)) {
            buffer = buffer.append(" = ").append(lines(fieldDecl.getValue()));
        }
        // After the child has been evaluated.
        declaredFields.add(field.getDeclaration());
        return buffer.append(";");
    }

    @Override
    public LineBuffer visitFieldReference(FieldReference fieldRef, None ctx) {
        Field field = fieldRef.getField();
        Field fieldDecl = field.getDeclaration();
        // If a field in the current class is referenced before its declared (lambdas), it must be fully qualified.
        // Otherwise, it's an illegal forward reference according to Javac.
        var currentClass = peekClass();
        boolean notDeclaredYet = currentClass != null && currentClass.getClazz() == fieldDecl.getDeclaringClass()
                                 && !declaredFields.contains(fieldDecl);
        LineBuffer buffer = of();
        if (fieldRef.getTarget() instanceof Nop) {
            ClassType type = field.getDeclaringClass();
            if (notDeclaredYet || importCollector.doesStaticFieldRequireQualifier(fieldRef.getTargetClassType(), field)) {
                buffer = of().append(importCollector.collect(type))
                        .append(".");
            }
        } else {
            if (notDeclaredYet || !(fieldRef.getTarget() instanceof LoadThis) || importCollector.isFieldHidden(fieldRef.getTargetClassType(), field)) {
                buffer = linesWrap(fieldRef.getTarget(), OperatorPrecedence.MEMBER_ACCESS)
                        .append(".");
            }
        }
        return buffer.append(fieldRef.getField().getName());
    }

    @Override
    public LineBuffer visitForEachLoop(ForEachLoop forEachLoop, None ctx) {
        LineBuffer buffer = of();
        buffer = buffer.append(blockHeader(forEachLoop.getBody()));
        buffer = buffer.add("for (");
        pushIndent();
        buffer = buffer.append(lines(forEachLoop.getVariable()));
        buffer = buffer.append(" : ");
        buffer = buffer.append(lines(forEachLoop.getIterator()));
        buffer = buffer.append(") {");
        buffer = buffer.join(indent(lines(forEachLoop.getBody())));
        popIndent();
        buffer = buffer.add("}");
        return buffer;
    }

    @Override
    public LineBuffer visitForLoop(ForLoop forLoop, None ctx) {
        LineBuffer buffer = of();
        buffer = buffer.append(blockHeader(forLoop.getBody()));
        buffer = buffer.add("for (");
        pushIndent();
        if (!(forLoop.getInitializer() instanceof Nop)) {
            buffer = buffer.append(lines(forLoop.getInitializer()));
        } else {
            lines(forLoop.getInitializer());
        }
        buffer = buffer.append("; ");
        buffer = buffer.append(lines(forLoop.getCondition())).append("; ");
        // Must be visited prior to increment block being visited to satisfy scope visitor constraints
        var body = indent(lines(forLoop.getBody()));
        buffer = buffer.append(lines(forLoop.getIncrement()));
        buffer = buffer.append(") {");
        buffer = buffer.join(body);
        popIndent();
        buffer = buffer.add("}");
        return buffer;
    }

    private LineBuffer printLambdaDecl(MethodDecl methodDecl, List<ParameterVariable> parameterVars) {
        functionStack.push(methodDecl);
        assert methodDecl.hasBody();
        BlockContainer body = methodDecl.getBody();

        // We can emit an expression lambda if we only contain a return.
        boolean expressionLambda = body.blocks.size() == 1
                                   && body.blocks.first().instructions.first() instanceof Return;

        declaredVariables.addAll(parameterVars);

        LineBuffer buffer = of();
        if (parameterVars.size() != 1) {
            buffer = buffer
                    .append("(")
                    .append(FastStream.of(parameterVars)
                            .map(this::getVariableName)
                            .join(", ")
                    )
                    .append(")");
        } else {
            buffer = buffer.append(getVariableName(parameterVars.getFirst()));
        }
        buffer = buffer.append(" -> ");
        if (expressionLambda) {
            Instruction insn = body.blocks.first().instructions.first().getFirstChild();
            if (insn instanceof Nop) {
                // Return is void, just print empty.
                buffer = buffer.append("{ }");
            } else {
                // Print return value.
                buffer = buffer.append(lines(insn));
            }
        } else {
            buffer = buffer.append("{");
            pushIndent();
            buffer = buffer.join(lines(body));
            popIndent();
            buffer = buffer.add("}");
        }
        functionStack.pop();
        return buffer;
    }

    @Override
    public LineBuffer visitMethodDecl(MethodDecl methodDecl, None ctx) {
        List<ParameterVariable> parameterVars = methodDecl.parameters
                .filter(e -> !e.isImplicit())
                .toLinkedList();
        var parent = methodDecl.getParentOrNull();
        boolean isLambda = parent != null && !(parent instanceof ClassDecl);
        if (isLambda) {
            return printLambdaDecl(methodDecl, parameterVars);
        }

        functionStack.push(methodDecl);
        Method method = methodDecl.getMethod();
        ClassType declaringClass = method.getDeclaringClass();
        AType returnType = method.getReturnType();

        boolean isAnnotationClass = declaringClass.getAccessFlags().get(AccessFlag.ANNOTATION);
        boolean isStaticInitializer = method.getName().equals("<clinit>");
        boolean isInstanceInitializer = declaringClass.getDeclType() == ClassType.DeclType.ANONYMOUS && method.getName().equals("<init>")
                && parameterVars.isEmpty();

        AnnotationSupplier annotationSupplier = method.getAnnotationSupplier();
        TypeAnnotationData methodReturnTypeAnnotations = annotationSupplier.getTypeAnnotations(TypeAnnotationLocation.METHOD_RETURN, returnType);

        LineBuffer buffer = of();
        buffer = buffer.join(importCollector.annotations(annotationSupplier.getAnnotations(methodReturnTypeAnnotations)));
        buffer = buffer.add("");
        if (!isStaticInitializer && !isInstanceInitializer) {
            if (!isAnnotationClass) {
                String access = AccessFlag.stringRep(method.getAccessFlags());
                if (declaringClass.isInterface()) {
                    if (method.isAbstract()) {
                        access = access.replace("public abstract ", "");
                    }
                    if (method.isStatic()) {
                        access = access.replace("public ", "");
                    } else {
                        access = access.replace("public", "default");
                    }
                }
                buffer = buffer.append(access);
            }

            String typeParam = typeParameters(method, annotationSupplier);
            buffer = buffer.append(typeParam);
            if (!typeParam.isEmpty()) {
                buffer = buffer.append(" ");
            }

            if (method.isConstructor()) {
                buffer = buffer.append(declaringClass.getName());
            } else {
                buffer = buffer.append(importCollector.collect(returnType, methodReturnTypeAnnotations));
                buffer = buffer.append(" ");
                buffer = buffer.append(method.getName());
            }

            // Record decls are the only thing which don't emit parameters.
            if (methodDecl.getTag() instanceof CompactConstructorTag) {
                declaredVariables.addAll(parameterVars);
            } else {
                buffer = buffer.append("(");
                boolean first = true;

                ClassType enclosingClass = declaringClass.getEnclosingClass().orElse(null);
                ReferenceType receiverType = method.isConstructor() ? enclosingClass != null ? TypeSystem.makeThisType(enclosingClass) : null : TypeSystem.makeThisType(declaringClass);
                if (receiverType != null) {
                    TypeAnnotationData receiverTypeAnnotations = annotationSupplier.getTypeAnnotations(TypeAnnotationLocation.METHOD_RECEIVER, receiverType);
                    if (!receiverTypeAnnotations.isEmpty()) {
                        if (method.isConstructor()) {
                            ClassType outer = declaringClass.getEnclosingClass()
                                    .orElseThrow(notPossible());
                            buffer = buffer.append(importCollector.collect(receiverType, receiverTypeAnnotations))
                                    .append(" ")
                                    .append(importCollector.collect(outer.getDeclaration()))
                                    .append(".");
                        } else {
                            buffer = buffer.append(importCollector.collect(receiverType, receiverTypeAnnotations))
                                    .append(" ");
                        }
                        buffer = buffer.append("this");
                        first = false;
                    }
                }

                List<Parameter> parameters = method.getParameters();
                for (int i = 0; i < parameterVars.size(); i++) {
                    ParameterVariable variable = parameterVars.get(i);
                    declaredVariables.add(variable);

                    if (!first) {
                        buffer = buffer.append(", ");
                    }
                    first = false;
                    AType type = variable.getType();
                    Parameter parameter = parameters.get(variable.pIndex);
                    buffer = buffer.append(AccessFlag.stringRep(parameter.getFlags()));
                    AnnotationSupplier pAnnSupplier = parameter.getAnnotationSupplier();
                    TypeAnnotationData typeAnnotations = pAnnSupplier.getTypeAnnotations(TypeAnnotationLocation.PARAMETER, type, parameter.getFormalIdx());
                    List<AnnotationData> annotations = pAnnSupplier.getAnnotations(typeAnnotations);
                    if (!annotations.isEmpty()) {
                        buffer = buffer.append(importCollector.annotations(annotations).joinOn(" "));
                        buffer = buffer.append(" ");
                    }
                    if (i == parameterVars.size() - 1 && type instanceof ArrayType && method.getAccessFlags().get(AccessFlag.VARARGS)) {
                        buffer = buffer.append(importCollector.collect(((ArrayType) type).getElementType(), typeAnnotations.step(TypeAnnotationData.Target.ARRAY_ELEMENT)));
                        buffer = buffer.append("...");
                    } else {
                        buffer = buffer.append(importCollector.collect(type, typeAnnotations));
                    }
                    buffer = buffer.append(" ");
                    buffer = buffer.append(getVariableName(variable));
                }
                buffer = buffer.append(")");
            }
            List<ReferenceType> exceptions = method.getExceptions();
            if (!exceptions.isEmpty()) {
                buffer = buffer.append(" throws ");
                for (int i = 0; i < exceptions.size(); i++) {
                    if (i != 0) {
                        buffer = buffer.append(", ");
                    }
                    ReferenceType type = exceptions.get(i);
                    buffer = buffer.append(importCollector.collect(type, annotationSupplier.getTypeAnnotations(TypeAnnotationLocation.METHOD_THROWS, type, i)));
                }
            }
        } else if (isStaticInitializer) {
            buffer = buffer.append("static");
        }
        if (!method.isAbstract()) {
            if (!isInstanceInitializer) {
                buffer = buffer.append(" ");
            }
            buffer = buffer.append("{");
            pushIndent();
            buffer = buffer.join(lines(methodDecl.getBody()));
            popIndent();

            buffer = buffer.add("}");
        } else {
            Object defaultAnnotationValue = method.getDefaultAnnotationValue();
            if (isAnnotationClass && defaultAnnotationValue != null) {
                buffer = buffer.append(" default ").append(importCollector.annotationValue(defaultAnnotationValue));
            }
            buffer = buffer.append(";");
        }

        functionStack.pop();
        return buffer;
    }

    @Override
    public LineBuffer visitIfInstruction(IfInstruction ifInsn, None ctx) {
        LineBuffer buffer = of("if (")
                .append(lines(ifInsn.getCondition()))
                .append(") {");
        pushIndent();
        if (!(ifInsn.getTrueInsn() instanceof Nop)) {
            buffer = buffer.join(indent(end(lines(ifInsn.getTrueInsn()), shouldEnd(ifInsn.getTrueInsn()))));
        } else {
            lines(ifInsn.getTrueInsn());
        }
        popIndent();
        if (!(ifInsn.getFalseInsn() instanceof Nop)) {
            Instruction falseInsn = ifInsn.getFalseInsn();
            if (falseInsn instanceof Block && falseInsn.getChildren().onlyOrDefault() != null && falseInsn.getFirstChild() instanceof IfInstruction) {
                buffer = buffer.add(indent("} else "))
                        .append(lines(ifInsn.getFalseInsn()).stripStart());
                return buffer;
            } else {
                buffer = buffer.add(indent("} else {"));
                pushIndent();
                buffer = buffer.join(indent(end(lines(ifInsn.getFalseInsn()), shouldEnd(ifInsn.getFalseInsn()))));
                popIndent();
            }
        }
        buffer = buffer.add(indent("}"));
        return buffer;
    }

    @Override
    public LineBuffer visitPostIncrement(PostIncrement postIncrement, None ctx) {
        return lines(postIncrement.getReference()).append(postIncrement.isPositive() ? "++" : "--");
    }

    @Override
    public LineBuffer visitInstanceOf(InstanceOf instanceOf, None ctx) {
        LineBuffer buffer = linesWrap(instanceOf.getArgument(), OperatorPrecedence.RELATIONAL)
                .append(" instanceof ");
        Instruction reference = instanceOf.getPattern();
        if (!(reference instanceof Nop)) {
            return buffer.append(lines(reference));
        }
        return buffer.append(importCollector.collect(instanceOf.getType()));
    }

    @Override
    public LineBuffer visitInvoke(Invoke invoke, None ctx) {
        MethodDecl currentFunc = functionStack.peek();
        LineBuffer buffer = of();
        Method method = invoke.getMethod();

        boolean isCtorCall = method.getName().equals("<init>") && currentFunc != null && currentFunc.getMethod().isConstructor();
        if (invoke.getKind() == InvokeKind.STATIC) {
            if (invoke.explicitTypeArgs || importCollector.doesStaticMethodRequireQualifier(invoke.getTargetClassType(), method.getName())) {
                buffer = of(importCollector.collect(invoke.getTargetClassType()))
                        .append(".");
            }
        } else if (!isCtorCall) {
            ClassType targetClass = invoke.getTargetClassType().getDeclaration();
            if (invoke.getKind() == InvokeKind.SPECIAL && ((ClassType)invoke.getTarget().getResultType()).getDeclaration() != targetClass) {
                // It's a call to our superclass.

                if (!(invoke.getTarget() instanceof LoadThis)) {
                    // Well, Target isn't LOAD_THIS, print it anyway, this won't recompile.
                    buffer = linesWrap(invoke.getTarget(), OperatorPrecedence.MEMBER_ACCESS).append(".");
                } else if (targetClass.isInterface()) {
                    buffer = buffer.append(importCollector.collect(targetClass)).append(".");
                }
                buffer = buffer.append("super")
                        .append(".");
            } else {
                if (invoke.explicitTypeArgs || !(invoke.getTarget() instanceof LoadThis) || importCollector.isMethodHidden(((LoadThis) invoke.getTarget()).getType().getDeclaration(), method.getName())) {
                    buffer = linesWrap(invoke.getTarget(), OperatorPrecedence.MEMBER_ACCESS);
                    buffer = buffer.append(".");
                }
            }
        }

        if (isCtorCall) {
            if (!(invoke.getTarget() instanceof LoadThis)) {
                buffer = lines(invoke.getTarget()).append(".");
            }
            if (InvokeMatching.matchConstructorInvokeSpecial(invoke, currentFunc.getMethod().getDeclaringClass()) != null) {
                buffer = buffer.append("this");
            } else {
                buffer = buffer.append("super");
            }
        } else {
            if (method instanceof ParameterizedMethod && invoke.explicitTypeArgs) {
                buffer = appendTypeArguments(((ParameterizedMethod) method).getTypeArguments(), buffer);
            }
            buffer = buffer.append(method.getName());
        }
        buffer = buffer.append(argList(invoke.getArguments()));
        return buffer;
    }

    @Override
    public LineBuffer visitInvokeDynamic(InvokeDynamic indy, None ctx) {
        return of("__invokedynamic ")
                .append(importCollector.collect(indy.bootstrapHandle.getDeclaringClass()))
                .append(".")
                .append(indy.bootstrapHandle.getName())
                .append(paren(FastStream.concat(FastStream.of(indy.name), FastStream.of(indy.bootstrapArguments))
                        .map(this::debugIndyBSMArg)
                        .fold(of(), (a, b) -> {
                            assert a != null;
                            if (!a.lines.isEmpty()) a = a.append(", ");
                            return a.append(b);
                        })
                ))
                .append(argList(indy.arguments));
    }

    @Override
    public LineBuffer visitMethodReference(MethodReference methodReference, None ctx) {
        Method method = methodReference.getMethod();
        Instruction target = methodReference.getTarget();
        LineBuffer buffer;
        if (target instanceof Nop) {
            if (method.isConstructor()) {
                return of(importCollector.collect(method.getDeclaringClass()) + "::new");
            }
            buffer = of(importCollector.collect(methodReference.getTargetClassType()) + "::");
        } else {
            buffer = linesWrap(target, OperatorPrecedence.MEMBER_ACCESS).append("::");
        }
        if (method instanceof ParameterizedMethod && methodReference.explicitTypeArgs) {
            buffer = appendTypeArguments(((ParameterizedMethod) method).getTypeArguments(), buffer);
        }
        return buffer.append(method.getName());
    }

    @Override
    public LineBuffer visitLdcBoolean(LdcBoolean ldcBoolean, None ctx) {
        return of(ldcBoolean.getValue() ? "true" : "false");
    }

    @Override
    public LineBuffer visitLdcChar(LdcChar ldcChar, None ctx) {
        // TODO escape unicode characters.
        return of("'" + EscapeUtils.escapeChar(ldcChar.getValue()) + "'");
    }

    @Override
    public LineBuffer visitLdcClass(LdcClass ldcClass, None ctx) {
        return of(importCollector.collect(ldcClass.getType()) + ".class");
    }

    @Override
    public LineBuffer visitLdcNumber(LdcNumber ldcNumber, None ctx) {
        String number = ldcNumber.getValue().toString();
        String prettyNumber = prettyPrintNumber(ldcNumber.getValue());

        boolean requiresSuffix = shouldPrintSuffix(ldcNumber, number);
        if (ldcNumber.getValue() instanceof Double) return of(requiresSuffix ? number : prettyNumber);
        if (ldcNumber.getValue() instanceof Float) return of(prettyNumber).append(requiresSuffix ? "F" : "");
        if (ldcNumber.getValue() instanceof Long) return of(number).append(requiresSuffix ? "L" : "");

        return of(number);
    }

    private String prettyPrintNumber(Number number) {
        String str = number.toString();

        if (!str.endsWith(".0")) return str;
        return str.substring(0, str.length() - 2);
    }

    private boolean shouldPrintSuffix(LdcNumber ldc, String numberStr) {
        AType type = ldc.getResultType();
        // Float exponents always require suffixes.
        if (type == PrimitiveType.FLOAT && numberStr.toUpperCase().contains("E")) {
            return true;
        }

        Instruction us = ldc;
        Instruction parent = ldc.getParentOrNull();
        // If we are in a negation, just look through
        if (matchNegation(parent) != null) {
            us = parent;
            parent = parent.getParent();
        }

        if (!(parent instanceof Binary binary)) return true;

        if (binary.getRight() == us) {
            // Right-hand side prioritizes printing suffix when both are an LdcNumber.
            return requiresSuffix(binary.getLeft(), type, ldc.getValue()) || unwrapNegation(binary.getLeft()) instanceof LdcNumber;
        }
        assert binary.getLeft() == us;

        // LdcNumber on left prints suffix if the right type is different,
        return requiresSuffix(binary.getRight(), type, ldc.getValue());
    }

    private boolean requiresSuffix(Instruction otherSide, AType ourType, Number ldc) {
        // Our side requires a suffix if the other side is of a different type.
        if (otherSide.getResultType() != ourType) return true;

        // Longs have some special rules and require suffix if they are larger than Integer.MAX_VALUE
        if (ourType == PrimitiveType.LONG && (ldc.longValue() > Integer.MAX_VALUE || ldc.longValue() < Integer.MIN_VALUE)) return true;

        // Floats are special and require suffix regardless of side
        // if the other side is not an LdcNumber and the float value is not whole.
        return ourType == PrimitiveType.FLOAT
               && !(otherSide instanceof LdcNumber)
               && (float) ldc.intValue() != ldc.floatValue();
    }

    private Instruction unwrapNegation(Instruction insn) {
        Instruction nValue = matchNegation(insn);
        return nValue != null ? nValue : insn;
    }

    @Override
    public LineBuffer visitLdcNull(LdcNull ldcNull, None ctx) {
        return of("null");
    }

    @Override
    public LineBuffer visitLdcString(LdcString ldcString, None ctx) {
        return of("\"" + EscapeUtils.escapeChars(ldcString.getValue()) + "\"");
    }

    @Override
    public LineBuffer visitLeave(Leave leave, None ctx) {
        // Don't print the natural exit.
        if (getNaturalExit(getOwningContainer(leave)) == leave) {
            return of();
        }

        if (!requiresLabelDefinition(leave)) {
            // Break on our own container.
            return of("break");
        }
        // Labeled break on another container.
        return of("break " + leave.getTargetContainer().getEntryPoint().getName());
    }

    @Override
    public LineBuffer visitLoad(Load load, None ctx) {
        return lines(load.getReference());
    }

    @Override
    public LineBuffer visitLoadThis(LoadThis loadThis, None ctx) {
        ClassDecl currentClass = peekClass();
        // TODO, Use the ImportCollector here. It needs to know about class declaration scopes.
        if (currentClass != null && loadThis.getType().getDeclaration() != currentClass.getClazz()) {
            return of(loadThis.getType().getName() + ".this");
        }
        return of("this");
    }

    @Override
    public LineBuffer visitLocalReference(LocalReference localRef, None ctx) {
        LineBuffer buffer = of();
        if (!scopeVisitor.isDeclared(localRef.variable)) {
            localRefDeclarations.add(localRef);
            AType type = localRef.variable.getType();
            AnnotationSupplier annotationSupplier = localRef.variable.getAnnotationSupplier();
            buffer = buffer.append(importCollector.collect(type, annotationSupplier.getTypeAnnotations(TypeAnnotationLocation.LOCAL_VARIABLE, type)))
                    .append(" ");
        }
        return buffer.append(getVariableName(localRef.variable));
    }

    @Override
    public LineBuffer visitLogicAnd(LogicAnd logicAnd, None ctx) {
        return linesWrap(logicAnd.getLeft(), OperatorPrecedence.LOGIC_AND).append(" && ").append(linesWrap(logicAnd.getRight(), OperatorPrecedence.LOGIC_AND));
    }

    @Override
    public LineBuffer visitLogicNot(LogicNot logicNot, None ctx) {
        return of("!").append(linesWrap(logicNot.getArgument(), OperatorPrecedence.UNARY));
    }

    @Override
    public LineBuffer visitLogicOr(LogicOr logicOr, None ctx) {
        return linesWrap(logicOr.getLeft(), OperatorPrecedence.LOGIC_OR).append(" || ").append(linesWrap(logicOr.getRight(), OperatorPrecedence.LOGIC_OR));
    }

    @Override
    public LineBuffer visitNewArray(NewArray newArray, None ctx) {
        LineBuffer buffer = of();
        if (!newArray.isInitializer) {
            AType rootElementType = newArray.getType();
            int numDims = 0;
            while (rootElementType instanceof ArrayType) {
                rootElementType = ((ArrayType) rootElementType).getElementType();
                numDims++;
            }
            buffer = buffer.append("new ").append(importCollector.collect(rootElementType));

            InstructionCollection<Instruction> indices = newArray.getIndices();
            for (Instruction index : indices) {
                buffer = buffer.append("[").append(lines(index)).append("]");
            }
            for (int i = indices.size(); i < numDims; i++) {
                buffer = buffer.append("[]");
            }
        } else {
            if (!isArrayInitializerTypeImplicit(newArray)) {
                buffer = buffer.append("new ").append(importCollector.collect(newArray.getType())).append(" ");
            }
            buffer = buffer.append(argList("{ ", newArray.getValues(), " }"));
        }

        return buffer;
    }

    private boolean isArrayInitializerTypeImplicit(NewArray newArray) {
        Instruction parent = newArray.getParent();
        AType parentType = parent.getResultType();
        // Nested array initializers must match the parent's element type.
        if (parent instanceof NewArray) {
            return ((ArrayType) parentType).getElementType().equals(newArray.getType());
        }
        // Stores, the Reference must be a LocalReference declaration.
        if (parent instanceof Store store) {
            // TODO what the fuck?
            if (localRefDeclarations.contains(store.getReference())) {
                // And the type must match exactly.
                return parentType.equals(newArray.getType());
            }
        }
        // Otherwise, only field declarations and an exact type match.
        return parent instanceof FieldDecl && parentType.equals(newArray.getType());
    }

    @Override
    public LineBuffer visitNew(New newInsn, None ctx) {
        Method method = newInsn.getMethod();
        IndexedInstructionCollection<Instruction> arguments = newInsn.getArguments();
        boolean printedTarget = false;
        Instruction target = newInsn.getTarget();
        LineBuffer buffer = of();
        if (target != null) {
            if (!(target instanceof LoadThis)) {
                buffer = buffer.append(linesWrap(target, OperatorPrecedence.MEMBER_ACCESS)).append(".");
                printedTarget = true;
            }
        }

        TypeAnnotationData extendsAnnotations = TypeAnnotationData.EMPTY;
        ClassType type = newInsn.getResultType();
        if (newInsn.hasAnonymousClassDeclaration()) {
            List<ClassType> superTypes = type.getDirectSuperTypes().toLinkedList();
            type = superTypes.getLast();

            // TODO Broken somehow, There is a weird INNER_TYPE path segment on the parameterization.
            //      only when the method is non-static.
//            AnnotationSupplier annotationSupplier = method.getDeclaration().getDeclaringClass().getAnnotationSupplier();
//            extendsAnnotations = annotationSupplier.getTypeAnnotations(TypeAnnotationLocation.CLASS_EXTENDS, type);
        }

        if (newInsn.explicitTypeArgs && method instanceof ParameterizedMethod) {
            buffer = appendTypeArguments(((ParameterizedMethod) method).getTypeArguments(), buffer);
        }

        buffer = buffer.append("new ")
                .append(importCollector.collect(type, extendsAnnotations, printedTarget, !newInsn.explicitClassTypeArgs && !newInsn.hasAnonymousClassDeclaration()));

        buffer = buffer.append(argList(arguments.filter(e -> e != target)));
        if (newInsn.hasAnonymousClassDeclaration()) {
            buffer = buffer.append(" ").append(lines(newInsn.getAnonymousClassDeclaration()));
        }
        return buffer;
    }

    @Override
    public LineBuffer visitMonitorEnter(MonitorEnter monitor, None ctx) {
        return of("__monitorenter ").append(lines(monitor.getArgument()));
    }

    @Override
    public LineBuffer visitMonitorExit(MonitorExit monitor, None ctx) {
        return of("__monitorexit ").append(lines(monitor.getArgument()));
    }

    @Override
    public LineBuffer visitNewObject(NewObject newObject, None ctx) {
        return of("__newobject " + importCollector.collect(newObject.getType()));
    }

    @Override
    public LineBuffer visitReturn(Return ret, None ctx) {
        // Don't print the natural exit.
        if (ret.getValue() instanceof Nop && getNaturalExit(getOwningContainer(ret)) == ret) {
            return of();
        }

        if (ret.getValue() instanceof Nop) {
            // Standard return.
            return of("return");
        }
        // Return with value
        return of("return ").append(lines(ret.getValue()));
    }

    @Override
    public LineBuffer visitBinary(Binary binary, None ctx) {
        OperatorPrecedence precedence = getPrecedence(binary);
        boolean wrapRight = !precedence.isLowerThan(getPrecedence(binary.getRight()));
        if (matchNegation(binary) != null) {
            return of("-").append(wrap(lines(binary.getRight()), wrapRight));
        }
        return linesWrap(binary.getLeft(), precedence).append(" " + binary.getOp().chars + " ").append(wrap(lines(binary.getRight()), wrapRight));
    }

    @Override
    public LineBuffer visitStore(Store store, None ctx) {
        return lines(store.getReference())
                .append(" = ")
                .append(linesWrap(store.getValue(), OperatorPrecedence.ASSIGNMENT));
    }

    @Override
    public LineBuffer visitSwitch(Switch switchInsn, None ctx) {
        LineBuffer buffer = blockHeader(switchInsn.getBody());
        buffer = buffer.join(of("switch (").append(lines(switchInsn.getValue())).append(") {"));
        pushIndent();
        buffer = buffer.join(lines(switchInsn.getBody()));
        popIndent();
        buffer = buffer.add("}");
        return buffer;
    }

    @Override
    public LineBuffer visitSwitchTable(SwitchTable switchTable, None ctx) {
        LineBuffer buffer = of();
        for (Instruction child : switchTable.sections) {
            buffer = buffer.join(lines(child));
        }
        return buffer;
    }

    @Override
    public LineBuffer visitSwitchSection(SwitchTable.SwitchSection switchSection, None ctx) {
        LineBuffer buffer = of();
        for (Instruction value : switchSection.values) {
            if (value instanceof Nop) {
                buffer = buffer.join(of("default:"));
            } else {
                buffer = buffer.join(of("case ").append(printSwitchValue(value)).append(":"));
            }
        }
        if (switchSection.explicitBlock) {
            buffer = buffer.append(" {");
        }
        pushIndent();
        buffer = buffer.join(indent(end(lines(switchSection.getBody()), shouldEnd(switchSection.getBody()))));
        popIndent();
        if (switchSection.explicitBlock) {
            buffer = buffer.add("}");
        }
        return buffer;
    }

    private LineBuffer printSwitchValue(Instruction value) {
        if (!(value instanceof FieldReference ref)) return lines(value);
        ClassType clazz = ref.getField().getDeclaringClass();

        if (ref.getField().getType() != clazz) return lines(value);
        if (!clazz.isEnum()) return lines(value);

        return of(ref.getField().getName());
    }

    @Override
    public LineBuffer visitSynchronized(Synchronized synchInsn, None ctx) {
        LineBuffer buffer = of(indent("synchronized("))
                .append(lines(synchInsn.getVariable()))
                .append(") {");
        pushIndent();
        buffer = buffer.join(lines(synchInsn.getBody()));
        popIndent();
        buffer = buffer.add(indent("}"));
        return buffer;
    }

    @Override
    public LineBuffer visitTernary(Ternary ternary, None ctx) {
        return linesWrap(ternary.getCondition(), OperatorPrecedence.TERNARY)
                .append(" ? ").append(linesWrap(ternary.getTrueInsn(), OperatorPrecedence.TERNARY))
                .append(" : ").append(linesWrap(ternary.getFalseInsn(), OperatorPrecedence.TERNARY));
    }

    @Override
    public LineBuffer visitThrow(Throw throwInsn, None ctx) {
        return of("throw ").append(lines(throwInsn.getArgument()));
    }

    private LineBuffer tryHeader(TryInstruction tryInsn, String header) {
        return blockHeader(tryInsn.getTryBody()).join(of(header));
    }

    @Override
    public LineBuffer visitTryCatch(TryCatch tryCatch, None ctx) {
        LineBuffer buffer = tryHeader(tryCatch, "try {");
        pushIndent();
        buffer = buffer.join(lines(tryCatch.getTryBody()));
        popIndent();
        buffer = buffer.add("}");
        for (TryCatch.TryCatchHandler handler : tryCatch.handlers) {
            buffer = buffer.append(lines(handler));
        }
        return buffer;
    }

    @Override
    public LineBuffer visitTryCatchHandler(TryCatch.TryCatchHandler catchHandler, None ctx) {
        LineBuffer buffer = of(" catch (");
        pushIndent();
        buffer = buffer.append(lines(catchHandler.getVariable()));
        buffer = buffer.append(") {");
        buffer = buffer.join(lines(catchHandler.getBody()));
        popIndent();
        buffer = buffer.add("}");
        return buffer;
    }

    @Override
    public LineBuffer visitTryFinally(TryFinally tryFinally, None ctx) {
        LineBuffer buffer = tryHeader(tryFinally, "try {");
        pushIndent();
        buffer = buffer.join(lines(tryFinally.getTryBody()));
        popIndent();
        buffer = buffer.add("} finally {");
        pushIndent();
        buffer = buffer.join(lines(tryFinally.getFinallyBody()));
        popIndent();
        buffer = buffer.add("}");
        return buffer;
    }

    @Override
    public LineBuffer visitTryWithResources(TryWithResources tryWithResources, None ctx) {
        pushIndent();
        LineBuffer buffer = tryHeader(tryWithResources, "try (")
                .append(lines(tryWithResources.getResource()))
                .append(") {")
                .join(lines(tryWithResources.getTryBody()))
                .add("}");
        popIndent();
        return buffer;
    }

    @Override
    public LineBuffer visitWhileLoop(WhileLoop whileLoop, None ctx) {
        LineBuffer buffer = blockHeader(whileLoop.getBody());
        buffer = buffer.add("while (").append(lines(whileLoop.getCondition())).append(") {");
        pushIndent();
        buffer = buffer.join(indent(lines(whileLoop.getBody())));
        popIndent();
        buffer = buffer.add("}");
        return buffer;
    }

    @Override
    public LineBuffer visitYield(Yield yield, None ctx) {
        return LineBuffer.of("yield ").append(lines(yield.getValue()));
    }
}
