package net.covers1624.coffeegrinder.bytecode.transform.transformers.statement;

import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import net.covers1624.coffeegrinder.bytecode.insns.tags.IIncTag;
import net.covers1624.coffeegrinder.bytecode.matching.AssignmentMatching;
import net.covers1624.coffeegrinder.bytecode.transform.StatementTransformContext;
import net.covers1624.coffeegrinder.bytecode.transform.StatementTransformer;
import org.jetbrains.annotations.Nullable;

import static net.covers1624.coffeegrinder.bytecode.matching.LoadStoreMatching.*;

/**
 * Created by covers1624 on 8/9/21.
 */
public class AssignmentExpressions implements StatementTransformer {

    @SuppressWarnings ("NotNullFieldNotInitialized")
    private StatementTransformContext ctx;

    @Override
    public void transform(Instruction statement, StatementTransformContext ctx) {
        this.ctx = ctx;
        Store store = matchStoreLocal(statement);
        if (store == null || store.getVariable().getKind() != LocalVariable.VariableKind.STACK_SLOT) return;

        // Match i++ and i--
        if (transformPostIncrementFromIinc(store)) {
            return;
        }

        if (transformPostIncrement(store)) {
            transform(store, ctx);
            return;
        }

        if (transformAssignmentExpression(store)) {
            transform(store, ctx);
            return;
        }

        tagPotentialIincInline(store);
    }

    private void tagPotentialIincInline(Store store) {
        // Tag ↓ with a reference to LOAD LOCAL i as potential inline
        //   COMPOUND_ASSIGNMENT.ADD (LOCAL i, LDC_INT(...))
        // > STORE (LOCAL s_1, LOAD LOCAL i)
        Load load = matchLoadLocal(store.getValue());
        if (load == null || load.getVariable().getKind() == LocalVariable.VariableKind.STACK_SLOT) return;

        CompoundAssignment iinc = matchIInc(store.getPrevSiblingOrNull(), load.getVariable());
        if (iinc == null) return;

        assert iinc.getTag() != null;
        ((IIncTag) iinc.getTag()).potentialInline = load;
    }

    private boolean transformPostIncrementFromIinc(Store store) {
        // STORE (LOCAL s_1, LOAD LOCAL i)
        // COMPOUND_ASSIGNMENT.ADD (LOCAL i, LDC_INT([1, -1]))
        // ->
        // STORE (LOCAL s_1, POST_INCREMENT (i[++, --]))

        // Make sure it loads a Local variable
        Load load = matchLoadLocal(store.getValue());
        if (load == null) return false;

        CompoundAssignment compoundAssignment = matchIInc(store.getNextSiblingOrNull(), load.getVariable());
        if (compoundAssignment == null) return false;

        LdcNumber ldc = (LdcNumber) compoundAssignment.getValue();
        if (ldc.intValue() != 1) return false;

        ctx.pushStep("Create post increment " + store.getVariable().getUniqueName());
        compoundAssignment.remove();
        load.replaceWith(new PostIncrement(load.getReference(), compoundAssignment.getOp() == BinaryOp.ADD));
        ctx.popStep();
        return true;
    }

    public boolean transformPostIncrement(Store store) {
        // STORE (LOCAL s_1, LOAD ref)
        // STORE (ref, NUMERIC.ADD (LOAD s_1, LDC_NUMBER([1, -1])))
        // ->
        // STORE (LOCAL s_1, POST_INCREMENT (ref, [++, --]))

        // Make sure it loads a Local variable
        if (!(store.getValue() instanceof Load load)) return false;

        if (!(store.getNextSiblingOrNull() instanceof Store store2)) return false;

        Reference reference = load.getReference();
        if (!equivalentRefs(reference, store2.getReference())) return false;
        Binary numeric = AssignmentMatching.matchStoreArgBinaryWithPossibleCast(store2);
        if (numeric == null || numeric.getOp() != BinaryOp.ADD && numeric.getOp() != BinaryOp.SUB) return false;
        if (matchLoadLocal(numeric.getLeft(), store.getVariable()) == null) return false;
        if (!(numeric.getRight() instanceof LdcNumber ldc)) return false;
        double value = ldc.getValue().doubleValue();
        if (Math.abs(value) != 1) return false;

        ctx.pushStep("Create post increment " + store.getVariable().getUniqueName());
        store2.remove();
        load.replaceWith(new PostIncrement(reference, (value > 0) == (numeric.getOp() == BinaryOp.ADD)));
        ctx.popStep();
        return true;
    }

    public boolean transformAssignmentExpression(Store store) {
        // when s_1 has more than 1 load...

        // STORE (LOCAL s_1, ...)
        // STORE (ref, LOAD s_1)
        // ->
        // STORE (LOCAL s_1, STORE (ref, ...))

        LocalVariable storeVariable = store.getVariable();

        // don't transform if the result wouldn't get used. That would just be round-about inlining
        if (storeVariable.getLoadCount() == 1) return false;

        if (!(store.getNextSiblingOrNull() instanceof Store store2)) return false;
        if (matchLoadLocal(store2.getValue(), storeVariable) == null) return false;

        ctx.pushStep("Create assignment expression " + storeVariable.getUniqueName());
        storeVariable.setType(store2.getResultType());
        store.getValue().replaceWith(new Store(store2.getReference(), store.getValue()));
        store2.remove();
        ctx.popStep();
        return true;
    }

    public static boolean equivalentRefs(Reference ref1, Reference ref2) {
        return switch (ref1) {
            case LocalReference r1 when ref2 instanceof LocalReference r2 -> r1.variable == r2.variable;
            case FieldReference r1 when ref2 instanceof FieldReference r2 -> equivalentRefs(r1, r2);
            case ArrayElementReference r1 when ref2 instanceof ArrayElementReference r2 -> equivalentRefs(r1, r2);
            default -> false;
        };
    }

    private static boolean equivalentRefs(FieldReference ref1, FieldReference ref2) {
        return equivalentFieldDescriptors(ref1.getField(), ref2.getField()) && equivalentSimpleValues(ref1.getTarget(), ref2.getTarget());
    }

    private static boolean equivalentRefs(ArrayElementReference ref1, ArrayElementReference ref2) {
        return equivalentSimpleValues(ref1.getArray(), ref2.getArray()) && equivalentSimpleValues(ref1.getIndex(), ref2.getIndex());
    }

    private static boolean equivalentSimpleValues(Instruction insn1, Instruction insn2) {
        return switch (insn1) {
            case Nop nop1 when insn2 instanceof Nop -> true;
            // might need to add lDC here
            case Load load1 when insn2 instanceof Load load2 -> //
                    load1.getReference() instanceof LocalReference lr1
                    && load2.getReference() instanceof LocalReference lr2
                    && lr1.variable == lr2.variable;
            default -> false;
        };
    }

    @Nullable
    public static CompoundAssignment matchIInc(@Nullable Instruction insn, LocalVariable variable) {
        if (!((insn instanceof CompoundAssignment compoundAssignment))) return null;
        if (!(insn.getTag() instanceof IIncTag)) return null;

        if (matchLocalRef(compoundAssignment.getReference(), variable) == null) return null;

        return compoundAssignment;
    }
}
