package net.covers1624.coffeegrinder.util.jvm;

import net.covers1624.coffeegrinder.util.resolver.JRTResolver;
import net.covers1624.coffeegrinder.util.resolver.Resolver;
import net.covers1624.jdkutils.JavaInstall;
import net.covers1624.jdkutils.utils.JavaPropExtractor;
import net.covers1624.quack.collection.FastStream;
import org.jetbrains.annotations.Nullable;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

/**
 * Created by covers1624 on 12/4/21.
 */
public class JVMUtils {

    private static final String JAVA_CLASS_PATH = "java.class.path";
    private static final String JAVA_BOOT_CLASS_PATH = "sun.boot.class.path";

    public static List<Resolver> getRuntimeJREClasspath() {
        return getJREClasspath(
                System.getProperty(JAVA_BOOT_CLASS_PATH),
                tryGetSysProp("java.home")
        );
    }

    public static JavaInstall getJavaInstall(Path executable) {
        JavaInstall install = JavaInstall.parse(executable);
        if (install == null) throw new RuntimeException("Unable to find valid Java installation at the provided Java executable.");
        return install;
    }

    public static List<Resolver> getTargetJREClasspath(JavaInstall javaInstall) {
        Map<String, String> props = JavaPropExtractor.extractProperties(
                JavaInstall.getJavaExecutable(javaInstall.javaHome, true),
                Collections.singletonList(JAVA_BOOT_CLASS_PATH)
        );
        if (props == null) throw new RuntimeException("Failed to extract java classpath information.");

        return getJREClasspath(
                props.get(JAVA_BOOT_CLASS_PATH),
                javaInstall.javaHome
        );
    }

    public static List<Resolver> getJREClasspath(@Nullable String bootClasspathProp, @Nullable Path javaHome) {
        // Java 8 and bellow use the 'sun.boot.class.path' system property.
        List<Path> paths = listProp(bootClasspathProp);
        if (paths != null) {
            return Resolver.findResolvers(paths);
        }

        // Java home required from here on (it should always exist, but may as well make sure)
        if (javaHome == null) throw new IllegalArgumentException("Unable to get classpath. Expected Boot Classpath or a Java Home.");
        if (Files.notExists(javaHome)) throw new IllegalArgumentException("Unable to get classpath. Expected Java Home to exist.");

        // Try JRT first, loads the JImage (if available) via the given
        // java's JRT file system.
        Throwable jrtException;
        try {
            return List.of(new JRTResolver(javaHome));
        } catch (IOException ex) {
            jrtException = ex;
        }

        // Fall back to raw jmods folder.
        // Most jdk's ship this alongside the jimage modules.
        Path jmods = javaHome.resolve("jmods");
        if (Files.isDirectory(jmods)) {
            return loadJMods(jmods);
        }

        // We tried chief.
        throw new IllegalArgumentException("Unable to get classpath. Java Home specified, but can't open a JRT FileSystem, or find any .jmod files.", jrtException);
    }

    public static List<Resolver> getRuntimeClasspath() {
        List<Path> paths = listSysProp(JAVA_CLASS_PATH);
        if (paths != null) {
            return Resolver.findResolvers(paths);
        }
        return Collections.emptyList();
    }

    private static @Nullable List<Path> listSysProp(String sysProp) {
        return listProp(System.getProperty(sysProp));
    }

    private static @Nullable List<Path> listProp(@Nullable String value) {
        if (value == null || value.isEmpty()) return null;

        return FastStream.of(value.split(File.pathSeparator))
                .distinct()
                .map(Paths::get)
                .filter(Files::exists)
                .toList();
    }

    private static @Nullable Path tryGetSysProp(String prop) {
        var value = System.getProperty(prop);
        if (value == null) return null;

        return Path.of(value);
    }

    private static List<Resolver> loadJMods(Path jmodsFolder) {
        try (Stream<Path> files = Files.list(jmodsFolder)) {
            return FastStream.of(files)
                    .map(Resolver::findResolver)
                    .toImmutableList();
        } catch (IOException e) {
            throw new IllegalStateException("Failed to list jmods directory.");
        }
    }
}
