package net.covers1624.coffeegrinder.source;

import net.covers1624.coffeegrinder.bytecode.InsnVisitor;
import net.covers1624.coffeegrinder.bytecode.Instruction;
import net.covers1624.coffeegrinder.bytecode.insns.Nop;
import net.covers1624.coffeegrinder.type.*;
import net.covers1624.coffeegrinder.util.None;
import net.covers1624.quack.collection.FastStream;
import org.jetbrains.annotations.Nullable;
import org.objectweb.asm.Type;

import java.util.List;

import static net.covers1624.coffeegrinder.source.LineBuffer.of;

/**
 * Created by covers1624 on 28/7/22.
 */
public abstract class AbstractSourceVisitor extends InsnVisitor<LineBuffer, None> {

    protected static final String INDENT = "    ";

    protected final ImportCollector importCollector;

    protected int indent;

    protected AbstractSourceVisitor(@Nullable TypeResolver typeResolver) {
        importCollector = new ImportCollector(typeResolver);
    }

    protected final void pushIndent() {
        indent++;
    }

    protected final void popIndent() {
        indent--;
    }

    protected LineBuffer lines(Instruction other) {
        return other.accept(this);
    }

    protected final LineBuffer indent(LineBuffer buffer) {
        return of(FastStream.of(buffer.lines)
                .map(this::indent)
                .toImmutableList()
        );
    }

    protected final String indent(String line) {
        // Don't indent empty lines.
        if (line.isEmpty()) return line;
        // Assume the line is already indented, do nothing.
        if (line.startsWith(" ")) return line;

        return INDENT.repeat(indent) + line;
    }

    protected boolean showImplicits() {
        return false;
    }

    protected final LineBuffer argList(Instruction... args) {
        return argList(FastStream.of(args));
    }

    protected final LineBuffer argList(FastStream<Instruction> args) {
        return argList("(", args, ")");
    }

    protected final LineBuffer argList(String prefix, FastStream<Instruction> args, String suffix) {
        LineBuffer buffer = of(prefix);
        boolean first = true;
        for (Instruction argument : args) {
            if (argument instanceof Nop && !showImplicits()) continue;
            if (!first) {
                buffer = buffer.append(", ");
            }
            first = false;
            buffer = buffer.append(lines(argument));
        }
        return buffer.append(suffix);
    }

    protected final String typeParameters(ITypeParameterizedMember member, AnnotationSupplier annotationSupplier) {
        List<TypeParameter> typeParameters = member.getTypeParameters();
        if (typeParameters.isEmpty()) return "";
        StringBuilder builder = new StringBuilder();
        builder.append("<");
        for (int i = 0; i < typeParameters.size(); i++) {
            TypeParameter param = typeParameters.get(i);
            if (i != 0) {
                builder.append(", ");
            }
            builder.append(importCollector.collectTypeParam(param, annotationSupplier));
        }
        builder.append(">");
        return builder.toString();
    }

    protected final LineBuffer appendTypeArguments(List<ReferenceType> args, LineBuffer buffer) {
        if (args.isEmpty()) return buffer;

        buffer = buffer.append("<");
        boolean first = true;
        for (ReferenceType typeArgument : args) {
            if (!first) {
                buffer = buffer.append(", ");
            }
            first = false;
            buffer = buffer.append(importCollector.collect(typeArgument));
        }
        buffer = buffer.append(">");
        return buffer;
    }

    protected final LineBuffer debugIndyBSMArg(Object obj) {
        if (obj instanceof Method meth) {
            return LineBuffer.of("METHOD ")
                    .append(importCollector.collect(meth.getDeclaringClass()))
                    .append(".")
                    .append(meth.getName());
        }
        if (obj instanceof Field field) {
            return LineBuffer.of("FIELD ")
                    .append(importCollector.collect(field.getDeclaringClass()))
                    .append(".")
                    .append(field.getName());
        }
        if (obj instanceof String || obj instanceof Type) {
            return LineBuffer.of("\"").append(EscapeUtils.escapeChars(obj.toString())).append("\"");
        }
        return LineBuffer.of(obj);
    }
}
