package net.covers1624.coffeegrinder.type.asm;

import net.covers1624.coffeegrinder.type.*;
import org.jetbrains.annotations.Nullable;
import org.objectweb.asm.TypePath;
import org.objectweb.asm.tree.TypeAnnotationNode;

import static net.covers1624.quack.util.SneakyUtils.notPossible;

/**
 * Created by covers1624 on 4/2/23.
 */
public class TypeAnnotationParser {

    public static void parseTypeAnnotation(TypeResolver resolver, @Nullable AType type, TypeAnnotationNode node, TypeAnnotationData rootNode) {
        TypeAnnotationData location = parseTypePath(node.typePath, 0, rootNode, type);
        node.accept(AnnotationParser.newVisitor(resolver, node.desc, e -> {
            // Records parse multiple locations into the same root node, filter to prevent duplicates.
            // We do not need to worry about repeatable annotations here as they will still be wrapped
            // into their container list annotation.
            if (!location.annotations.contains(e)) {
                location.annotations.add(e);
            }
        }));
    }

    private static TypeAnnotationData parseTypePath(@Nullable TypePath path, int step, TypeAnnotationData node, @Nullable AType type) {
        // Type should only be null for TYPE_ARGUMENT TypeAnnotationNode's. Which should also not have a path.
        if (path == null && type == null) return node;
        assert type != null;

        if (type instanceof ArrayType) {
            // Because of a 'bug' in java type paths, the order of selection for nested arrays is reversed.
            // The innermost element type is still at the appropriate depth, but the path then enumerates outwards from the innermost to outermost array
            // @Ann char @Ann [] @Ann [] @Ann []
            // ^ [[[     ^none   ^ [     ^[[

            // in order to determine how to evaluate the type path, we need to know how deep the array is
            int arrayDepth = arrayDepth(type);

            // consume all ARRAY_ELEMENT parts of the type path in one go
            int depth = 0;
            while (path != null && step < path.getLength() && path.getStep(step) == TypePath.ARRAY_ELEMENT) {
                depth++;
                step++;
            }

            // if the path depth doesn't match the array depth, reverse the depth to match the type-system
            if (depth < arrayDepth) {
                depth = arrayDepth - 1 - depth;
            }

            while (depth-- > 0) {
                node = node.addStep(TypeAnnotationData.Target.ARRAY_ELEMENT);
                type = ((ArrayType) type).getElementType();
            }
        }

        // Javac emits/models these backwards, inner of an outer. We model outer of an inner.
        if (type instanceof ClassType clazz && TypeSystem.isConstructedViaTargetInstance((ClassType) type)) {
            int parameterizedDepth = instancedDepth(clazz);

            int depth = 0;
            while (path != null && step < path.getLength() && path.getStep(step) == TypePath.INNER_TYPE) {
                depth++;
                step++;
            }
            depth = parameterizedDepth - depth;
            while (depth-- > 0) {
                node = node.addStep(TypeAnnotationData.Target.OUTER_TYPE);
                clazz = clazz.getEnclosingClass().orElseThrow(notPossible());
                type = clazz;
                assert type != null;
            }
        }

        if (path == null || path.getLength() == step) {
            return node;
        }
        TypeAnnotationData.Target stepType = TypeAnnotationData.Target.values()[path.getStep(step)];
        int stepArg = stepType == TypeAnnotationData.Target.TYPE_ARGUMENT ? path.getStepArgument(step) : 0;

        switch (stepType) {
            case WILDCARD_BOUND:
                if (type instanceof WildcardType wildcardType) {
                    type = wildcardType.isSuper() ? wildcardType.getLowerBound() : wildcardType.getUpperBound();
                }
                break;
            case TYPE_ARGUMENT:
                if ((type instanceof ParameterizedClass pClass)) {
                    type = pClass.getTypeArguments().get(stepArg);
                }
                break;
        }
        return parseTypePath(path, step + 1, node.addStep(stepType, stepArg), type);
    }

    private static int arrayDepth(AType type) {
        return type instanceof ArrayType ? 1 + arrayDepth(((ArrayType) type).getElementType()) : 0;
    }

    private static int instancedDepth(ClassType clazz) {
        return TypeSystem.isConstructedViaTargetInstance(clazz) ? 1 + instancedDepth(clazz.getEnclosingClass().orElseThrow(notPossible())) : 0;
    }
}
