package net.covers1624.coffeegrinder.debug;

import net.covers1624.coffeegrinder.bytecode.VariableLivenessGraph;
import net.covers1624.coffeegrinder.bytecode.flow.ControlFlowNode;
import net.covers1624.coffeegrinder.util.resolver.ClassResolver;
import net.covers1624.quack.collection.FastStream;
import net.covers1624.quack.util.SneakyUtils;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.nio.file.Path;
import java.util.ServiceLoader;

/**
 * An abstract interface for extended debugging.
 * <p>
 * Created by covers1624 on 15/9/25.
 */
public interface Debugger {

    /**
     * A static instance of the debugger, if available.
     */
    @Nullable Debugger DEBUGGER = FastStream.of(ServiceLoader.load(Debugger.class))
            .firstOrDefault();

    /**
     * Generate a GraphViz png for the given {@link ControlFlowNode} graph.
     *
     * @param cfNode The node to generate the graph for.
     * @param dest   The file to write to.
     * @throws IOException Any IO exception whilst generating or writing the graph.
     */
    void writeControlFlowGraph(ControlFlowNode cfNode, Path dest) throws IOException;

    /**
     * Generate a GraphViz Dot graph string for the given {@link ControlFlowNode} graph.
     *
     * @param cfNode The node to generate the graph for.
     * @return The Dot graph string.
     */
    String buildControlFlowGraphDot(ControlFlowNode cfNode);

    /**
     * Generate a GraphViz png for the given {@link VariableLivenessGraph}.
     *
     * @param vlGraph The variable liveness graph.
     * @param dest    The file to write to.
     * @throws IOException Any IO exception whilst generating or writing the graph.
     */
    void writeVariableLivenessGraph(VariableLivenessGraph vlGraph, Path dest) throws IOException;

    /**
     * Generate a GraphViz Dot graph string for the given {@link VariableLivenessGraph}.
     *
     * @param vlGraph The variable liveness graph.
     * @return The Dot graph string.
     */
    String buildVariableLivenessGraphDot(VariableLivenessGraph vlGraph);

    /**
     * Start an interactive http based web debugger.
     *
     * @param ipAndPort The ip and port to bind to.
     * @param resolver  The {@link ClassResolver} to use for target, library, and jdk information.
     */
    void startDebugger(String ipAndPort, ClassResolver resolver);

    /**
     * Try to run the given action, if the debugger is available,
     * fire the action with it. Otherwise, log the attempt and do nothing.
     *
     * @param action The action.
     */
    static void tryIfPresent(SneakyUtils.ThrowingConsumer<Debugger, Throwable> action) {
        if (DEBUGGER != null) {
            try {
                action.accept(DEBUGGER);
            } catch (Throwable ex) {
                Internal.LOGGER.error("Error running debugger action.", ex);
            }
            return;
        }
        Internal.LOGGER.warn("Debugger is not present.", new Throwable());
    }

    class Internal {

        private static final Logger LOGGER = LoggerFactory.getLogger(Debugger.class);
    }
}
