package net.covers1624.coffeegrinder.bytecode;

import net.covers1624.coffeegrinder.DecompilerSettings;
import net.covers1624.coffeegrinder.bytecode.insns.*;
import net.covers1624.coffeegrinder.bytecode.transform.*;
import net.covers1624.coffeegrinder.bytecode.transform.transformers.*;
import net.covers1624.coffeegrinder.bytecode.transform.transformers.generics.GenericTransform;
import net.covers1624.coffeegrinder.bytecode.transform.transformers.statement.*;
import net.covers1624.coffeegrinder.debug.Step;
import net.covers1624.coffeegrinder.debug.Step.StepContextType;
import net.covers1624.coffeegrinder.debug.Stepper;
import net.covers1624.coffeegrinder.type.*;
import net.covers1624.coffeegrinder.type.asm.AsmClass;
import net.covers1624.coffeegrinder.type.asm.AsmField;
import net.covers1624.coffeegrinder.util.asm.OrderedTextifier;
import org.objectweb.asm.tree.FieldNode;

import java.util.List;

/**
 * Created by covers1624 on 7/5/21.
 */
public class ClassProcessor {

    private final TypeResolver typeResolver;
    private final ClassType clazz;
    private final DecompilerSettings settings;

    private final List<MethodTransformer> methodTransforms;
    private final List<ClassTransformer> classTransforms;
    private final List<TopLevelClassTransformer> topLevelClassTransforms;

    public ClassProcessor(TypeResolver typeResolver, ClassType clazz, DecompilerSettings settings) {
        this.typeResolver = typeResolver;
        this.clazz = clazz;
        this.settings = settings;
        methodTransforms = List.of(
                new TryCatches(),
                new J11TryWithResourcesTransform(typeResolver),
                new IntegerConstantInference(),
                MethodBlockTransform.of("Switches",
                        new SwitchDetection()
                ),
                MethodBlockTransform.of("Loops",
                        new LoopDetection()
                ),
                new DetectExitPoints(),
                // SwitchExpressions must run VERY early as it relies on matching stack store + leave patterns,
                // if we leave it till after ConditionDetection, it will merge exits which may be yields. This would
                // be much harder to match, but also a pain to unwind as we can't easily unwrap else's and such.
                new SwitchExpressions(),
                MethodBlockTransform.of("Conditions and Statements",
                        new ConditionDetection(),
                        BlockStatementTransform.of("Statements",
                                new GeneratedNullChecks(),
                                new TernaryExpressions(),
                                new BooleanLogicYields(),
                                new SwitchOnString(),
                                new NewObjectTransform(),
                                new AssignmentExpressions(),
                                new AccessorTransforms(),
                                // Anything which may open up new inlining opportunities should be placed before here if possible.
                                new Inlining(),
                                new ArrayInitializers(), // Requires inlining to match. Requests rerun because array with initializer may be inlinable
                                new StringConcat(typeResolver),
                                new ExpressionTransforms(), // Simplifies statements after inlining
                                new CompoundAssignments()
                        ),
                        new LabelledBlocks()
                ),
                new SwitchInlining(),
                new HighLevelLoops(typeResolver),
                new ExitPointCleanup(),
                new LegacyTryWithResourcesTransform(typeResolver),
                new SynchronizedTransform(),
                new VariableDeclarations(),
                new CompoundAssignments() // Rerun after VariableDeclarations has merged local variables.
        );
        classTransforms = List.of(
                new FieldInitializers(),
                new Lambdas(typeResolver),
                new LocalClasses(),
                new InnerClasses(),
                new EnumClasses(),
                new AssertTransform(typeResolver),
                new RecordTransformer(typeResolver),
                new NumericConstants(typeResolver)
        );
        topLevelClassTransforms = List.of(
                new SwitchOnEnum(),
                new SwitchCleanup(),
                new SyntheticCleanup(),
                new ImplicitConstructorCleanup(),
                new GenericTransform()
        );
    }

    public ClassDecl process(Stepper stepper) {
        TransformContextBase baseCtx = new TransformContextBase(stepper, typeResolver, settings, clazz.getClassVersion());

        ClassDecl decl = new ClassDecl(clazz);
        stepper.pushContext(Stepper.ContextSupplier.ofInsn(typeResolver, decl));
        stepper.pushStepWithContent(clazz.getName(), StepContextType.CLASS, () -> OrderedTextifier.textify(((AsmClass) clazz).getNode().getNode()));
        decl.addRef();

        for (ClassType nestedClass : clazz.getNestedClasses().reversed()) {
            ClassProcessor processor = new ClassProcessor(typeResolver, nestedClass, settings);
            decl.members.add(processor.process(stepper));
        }

        for (Field field : clazz.getFields()) {
            FieldNode node = ((AsmField) field).getNode();
            FieldDecl insn = new FieldDecl(field);
            if (node.value != null) {
                insn.setValue(parseFieldConstant(insn.getResultType(), node.value));
            }
            decl.members.add(insn);
        }

        for (Method method : clazz.getMethods()) {
            MethodDecl methodDecl = InstructionReader.parse(typeResolver, method);

            stepper.pushContext(Stepper.ContextSupplier.ofInsn(typeResolver, methodDecl));
            stepper.pushStep(methodDecl.getMethod().getName(), StepContextType.METHOD);

            // Apply invariants from the reader. Importantly done after step push, so we can see in debugger.
            InvariantVisitor.checkInvariants(methodDecl);

            processMethod(baseCtx, methodDecl);

            decl.members.add(methodDecl);
            methodDecl.releaseRef(); // Ref added during parsing

            stepper.popStep();
            stepper.popContext();
        }

        decl.onParsed();

        processClass(baseCtx, decl);

        if (clazz.getDeclType() == ClassType.DeclType.TOP_LEVEL) {
            processTopLevelClass(baseCtx, decl);
        }

        stepper.popContext();
        stepper.popStep();
        return decl;
    }

    private void processMethod(TransformContextBase baseCtx, MethodDecl decl) {
        MethodTransformContext methodCtx = new MethodTransformContext(baseCtx, decl);
        for (MethodTransformer t : methodTransforms) {
            try (Step ignored = methodCtx.pushStep(t.getName(), t.stepType())) {
                t.transform(decl, methodCtx);
                InvariantVisitor.checkInvariants(decl);
            } catch (Throwable e) {
                methodCtx.except(e);
            }
        }
    }

    private void processClass(TransformContextBase baseCtx, ClassDecl decl) {
        ClassTransformContext ctx = new ClassTransformContext(baseCtx, decl);
        for (ClassTransformer t : classTransforms) {
            try (Step ignored = ctx.pushStep(t.getName(), t.stepType())) {
                t.transform(decl, ctx);
                InvariantVisitor.checkInvariants(decl);
            } catch (Throwable e) {
                ctx.except(e);
            }
        }
    }

    private void processTopLevelClass(TransformContextBase baseCtx, ClassDecl decl) {
        ClassTransformContext ctx = new ClassTransformContext(baseCtx, decl);
        for (TopLevelClassTransformer t : topLevelClassTransforms) {
            try (Step ignored = ctx.pushStep(t.getName(), t.stepType())) {
                t.transform(decl, ctx);
                InvariantVisitor.checkInvariants(decl);
            } catch (Throwable e) {
                ctx.except(e);
            }
        }
    }

    private Instruction parseFieldConstant(AType desiredType, Object obj) {
        if (desiredType == PrimitiveType.BOOLEAN) {
            return new LdcBoolean(((Integer) obj) == 1);
        }
        if (obj instanceof String str) {
            return new LdcString(typeResolver.resolveClass(TypeResolver.STRING_TYPE), str);
        }
        if (obj instanceof Number num) {
            if (desiredType == PrimitiveType.CHAR) {
                int intValue = num.intValue();
                assert intValue >= Character.MIN_VALUE && intValue <= Character.MAX_VALUE;
                return new LdcChar((char) intValue);
            }
            return new LdcNumber(num);
        }
        throw new IllegalArgumentException("Unknown field constant type: " + obj.getClass().getName());
    }
}
